/*
 *	C storage allocator
 *	(circular first fit strategy)
 *
 *	Maintains a monotonically increasing chain of pointers.
 *	If the memory between any two pointers is busy, the first pointer
 *	is so marked, else the memory is considered free.
 *	It is possible to sbrk directly, and 'free' this memory.
 *	Free chunks are ONLY coalesced during allocation.
 */ 

#define BLOK 512	/* sbrk for this many words each time */
#define BUSY 01

char *allocs[2]		/*initial empty arena*/
{
	&allocs[1],
	&allocs[0]
};
struct
{
	int word;
};
char **allocp &allocs[1];	/* start point for searches */
char **alloct &allocs[1];	/* top of arena (last cell) */ 

alloc(nbytes)
{
	register int nwords;
	register char **p, **q;
	char **t;


	allocs[0].word =| BUSY;	/* static initialization */ 
	allocs[1].word =| BUSY;

	nwords = (nbytes+3)/2;	/* round up num. bytes, plus one for pointer */
	p = allocp;
	for (;;)
	{
		/*
		 *	chain along list of pointers testing each free area
		 */
		do
		{
			if ((p->word & BUSY) == 0)	/* free */
			{
				while (((q = *p)->word & BUSY) == 0)
					*p = *q;		/* coalesce */
				if ((&p[nwords] >= p)		/* end has not wrapped around */
				    && (q >= &p[nwords]))	/* free area found is big enough */
				{
					/*
					 *   if necessary, break the area found
					 *  into two smaller bits (first
					 *  of which is the area requested)
					 */
					allocp = &p[nwords];	/* end of first bit */
					if (q > allocp)		/* must break */
						*allocp = q;
					*p = allocp.word|BUSY;
					return(p+1);
				}
			}
			q = p;
			p = p->word & ~BUSY;	/* chain along */
		}
		while (q >= allocp || p < allocp);

		/*
		 *	we have wrapped right around without finding
		 *	an adequate area - so must sbrk for more core
		 */

		t = sbrk(BLOK * 2);
		if (t.word == -1)
			return(-1);
		*alloct = t;

		/*
		 *	is this area contiguous with the last core break?
		 *	if so, set it up so that the next alloc will
		 *	coalesce them; if not, pretend the intervening
		 *	area is busy. NOTE: this implies that you may
		 *	break directly, and may 'free' this area.
		 */

		if (t != alloct+1)
			alloct->word =| BUSY;
		alloct = (*t = &t[BLOK]-1);
		*alloct = allocs;
		alloct->word =| BUSY;
	}
}

free(p)
register char **p;
{
	register char **r, **s;


	/*
	 *	consistency check - is it on the list?
	 */

	r = allocs;
	do
	{
		s = r->word & ~BUSY;
	}
	while (s && (s != p-1) && ((r = s) != allocs));
	if (s != p-1)
	{
		abort("free error");
	}

	/*
	 *	yes - turn off busy bit and reset allocp
	 *	(probably quite pointless)
	 */

	allocp = s;
	allocp->word =& ~BUSY;
}
