/*
 * New-new librarian for fortran linker
 *		Craig McGregor
 */

#include <stdio.h>

typedef unsigned short		rad50;
typedef struct	{
	rad50			rad50x[2];
}				rad50n;
#define	RADEQ(a,b)	((a).rad50x[0]==(b).rad50x[0] && (a).rad50x[1]==(b).rad50x[1])
#define	RADZERO(a)	{ (a).rad50x[0] = 0; (a).rad50x[1] = 0; }

#define	GOBBLE		512
#define	BINSIZ		132
#define	NIL		0

/*
 * possible args for "getitem"
 */
#define	COMMAND		0
#define	ARGUMENT	1

FILE				*ofile;
char				obuf[BUFSIZ];

/*
 * Reference structure
 *	undefined/implied
 */
struct	ref {
	rad50n			*r_name;
	struct ref		*r_next;
};

/*
 * List of global symbols
 */
struct	glist {
	rad50n			gl_nam;
	struct glist		*gl_nxt;
};

/*
 * File name + buffer?
 */
struct	file {
	struct fbuf		*ff_fb;
	char			ff_fnm[66];
};
struct	fbuf {
	FILE			*ff_io;
	struct fbuf		*ff_next;
	char			ff_buf[BUFSIZ];
};

/*
 * Module
 */
struct	mod {
	rad50n			m_nam;
	rad50n			m_ver;
	int			m_action;
#define	ACT_INF		001		/* new input file */
#define	ACT_CLS		002		/* close file */
#define	ACT_PAS		004		/* pass to new library */
#define	ACT_XTR		010		/* extract this module */
	struct file		*m_file;
	struct glist		*m_dgbl;	/* define globals */
	struct glist		*m_rgbl;	/* reference globals */
	struct mod		*m_next;
};

/*
 * Binary record stuff
 */
unsigned int			bcount;
char				*bpoint;
char				binbuf[BINSIZ];
#define	GSD		01
#define	ESD		02
#define	TXT		03
#define	RLD		04
#define	ISD		05
#define	EMOD		06
#define	LMOD		07
#define	PSD		017
#define	SDR		022
#define	EDR		021

/*
 * Global-symbol-directory entry from input records
 */
struct	gsd {
	rad50n			gsd_nm;
	char			g_flags;
#define	DEF		010
	char			g_type;
#define	MDN		0
#define	CSN		1
#define	ISN		2
#define	TRA		3
#define	GSN		4
#define	PSN		5
#define	PVI		6
	short			g_v;
}				gsd;

struct mod			*command;	/* from library #1 */
struct file			*libfile;	/* library #1 */
char				tmpfile[] = "libr1.tmp";
struct fbuf			*froot;		/* root of file buffers */

/* for processing arguments */
unsigned int			index;
unsigned int			iargc;
char				**iargv;

/* control flags */
int				listf;		/* produce listing */
int				creff;		/* cross-reference */
int				actionf;	/* new library/extraction */
int				extractf;	/* do extractions */
int				debugf;		/* list core usage */

char				*memlow;
char				*memhigh;
char				*memmax;

extern struct mod		*makelist();
extern char			*rawitem();
extern struct mod		**find();
extern char			*malloc();
extern struct file		*getfile();
extern char			*radixout();
extern rad50n			*loc();

main(argc, argv)
unsigned int			argc;
char				*argv[];
{
	setbuf(stderr, NULL);

	iargc = argc - 1;
	iargv = &argv[1];

	memlow = sbrk(GOBBLE);
	if (memlow == -1) {
		fprintf(stderr, "Core allocation failure.\n");
		xerror();
	}
	memhigh = memmax = memlow + GOBBLE;

	pass1();

	if (command == NIL) {
		fprintf(stderr, "Null library\n");
		xerror();
	}

	if (listf) {
		register struct mod		*mp;
		register struct glist		*gp;
		int				lcount;

		printf("\n\nContents of library %s", libfile->ff_fnm);
		printf("\nModule\tIdent\tDeclares\n\n");
		for (mp = command; mp != NIL; mp = mp->m_next)
			if (mp->m_action & ACT_PAS) {
				printf("%s\t", radixout(&mp->m_nam));
				printf("%s", radixout(&mp->m_ver));
				lcount = 6;
				for (gp = mp->m_dgbl; gp != NIL; gp = gp->gl_nxt) {
					if (--lcount < 0) {
						printf("\n\t");
						lcount = 6;
					}
					printf("\t%s", radixout(&gp->gl_nam));
				}
				printf("\n\n");
			}
	}

	if (creff)
		crefout();

	if (actionf) {
		if ((ofile = fopen(tmpfile, "w")) == NULL) {
			fprintf(stderr, "Cannot create temporary file : %s\n", tmpfile);
			xerror();
		}
		setbuf(ofile, obuf);

		pass2();

		fclose(ofile);
	} else if (extractf)
		pass2();

	if (debugf)
		fprintf(stderr, "Maxmem = %o\n", memmax);

	if (actionf)
		execl("/bin/mv", "libr.mv", tmpfile, libfile->ff_fnm, 0);

	return(0);
}

xerror()
{
	exit(1);
}

/*
 * Get next item from argument stream
 */
char *
rawitem()
{
	if (index == iargc)
		return(NIL);
	return(iargv[index++]);
}

char *
getitem(itemcode)
int				itemcode;
{
	register char			*r;
	static char			*peeks;

	if (peeks == NIL)
		peeks = rawitem();
	if (*peeks=='-' && itemcode==ARGUMENT)
		return(NIL);
	r = peeks;
	peeks = NIL;
	return(r);
}

struct file *
getfile()
{
	register char			*s1;
	register char			*s2;
	register unsigned int		cc;
	struct file			*ffp;

	if ((s1 = getitem(ARGUMENT)) == NIL)
		return(NIL);
	ffp = malloc(sizeof(struct file));
	s2 = ffp->ff_fnm;
	for (cc = 0; *s1!='\0' && cc<65; cc++)
		*s2++ = *s1++;
	*s2++ = '\0';
	return(ffp);
}

/*
 * Convert ascii string to radix50
 */
rad50n *
getarg()
{
	register char			*s;
	static rad50n			radraw;
	register rad50			r50;
	unsigned int			cc;
	unsigned int			w;
	char				c;
	char				*ss;

	ss = s = getitem(ARGUMENT);

	if (s == NIL)
		return(NIL);

	for (w = 0; w < 2; w++) {
		r50 = 0;
		for (cc = 0; cc < 3; cc++) {
			c = *s++;
			r50 *= 050;
			if (c == '\0') {
				/* backtrack so we finish filling name */
				s--;
				continue;
			}
			if (c == ' ')
				continue;
			if (c>='a' && c<='z') {
				r50 += c-'a' + 1;
				continue;
			}
			if (c == '$') {
				r50 += 27;
				continue;
			}
			if (c == '.') {
				r50 += 28;
				continue;
			}
			if (c == '_') {
				r50 += 29;
				continue;
			}
			if (c>='0' && c<='9') {
				r50 += c-'0' + 30;
				continue;
			}
			fprintf(stderr, "%s : bad radix 50\n", ss);
			xerror();
		}
		radraw.rad50x[w] = r50;
	}
	return(&radraw);
}

/*
 * Output radix50 name
 */
char *
radixout(w)
rad50n				*w;
{
	register unsigned int		i;
	register rad50			c1;
	static char			ss[] =  " abcdefghijklmnopqrstuvwxyz$._0123456789";
	static char			sout[7];
	register char			*s;

	s = sout;
	for (i = 0; i < 2; i++) {
		c1 = w->rad50x[i];
		*s++ = ss[c1 / (050*050)];
		*s++ = ss[(c1 / 050)%050];
		*s++ = ss[c1 % 050];
	}
	for (; s > sout; s--)
		if (s[-1] != ' ')
			break;
	*s = '\0';
	return(sout);
}

pass1()
{
	register struct mod		*mp;
	register struct mod		**mpp;
	rad50n				*whmod;
	rad50n				*whsym;
	struct file			*whfile;
	char				*cmd;

	if ((libfile = getfile()) == NIL) {
		fprintf(stderr, "Cannot get library name\n");
		xerror();
	}
	command = makelist(libfile);

	while ((cmd = getitem(COMMAND)) != NIL) {
		if (*cmd != '-') {
			fprintf(stderr, "%s\n", cmd);
			xerror();
		}
		switch (cmd[1]) {

		default:
			fprintf(stderr, "%s : unrecognized command\n", cmd);
			xerror();
			break;

		case 'l':
			listf = 1;
			break;

		case 'c':
			creff = 1;
			listf = 1;
			break;

		case 'z':
			debugf = 1;
			break;

		case 'g':
			actionf = 1;
			if ((whmod = getarg()) == NIL) {
				fprintf(stderr, "No module name for GLOBAL\n");
				xerror();
			}
			if ((mp = *find(&command, whmod)) == NIL) {
				fprintf(stderr, "%s : not found\n", radixout(whmod));
				xerror();
			}
			while ((whsym = getarg()) != NIL) {
				register struct glist		**gpp;
				register struct glist		*gp;

				for (gpp = &mp->m_dgbl; (gp = *gpp) != NIL; gpp = &gp->gl_nxt)
					if (RADEQ(gp->gl_nam, *whsym))
						break;
				if (gp == NIL) {
					fprintf(stderr, "%s : symbol not found\n", radixout(whmod));
					xerror();
				}
				*gpp = gp->gl_nxt;
			}
			break;

		case 'd':
			actionf = 1;
			while ((whmod = getarg()) != NIL) {
				if ((mp = *find(&command, whmod)) == NIL) {
					fprintf(stderr, "%s : module not found\n", radixout(whmod));
					xerror();
				}
				mp->m_action &= ~ACT_PAS;
			}
			break;

		case 'a':
			actionf = 1;
			while ((whfile = getfile()) != NIL) {
				if ((mp = makelist(whfile)) == NIL) {
					fprintf(stderr, "%s : file empty\n", whfile->ff_fnm);
					xerror();
				}
				for (mpp = &command; *mpp != NIL; mpp = &(*mpp)->m_next);
				*mpp = mp;
			}
			break;

		case 'i':
			actionf = 1;
			if ((whmod = getarg()) == NIL) {
				fprintf(stderr, "No module name for insert\n");
				xerror();
			}
			if (*(mpp = find(&command, whmod)) == NIL) {
				fprintf(stderr, "%s : cannot find module\n", radixout(whmod));
				xerror();
			}
			while ((whfile = getfile()) != NIL) {
				struct mod			*omp;

				if ((mp = makelist(whfile)) == NIL) {
					fprintf(stderr, "%s : cannot open\n", whfile->ff_fnm);
					xerror();
				}
				omp = *mpp;
				*mpp = mp;
				for (; mp->m_next != NIL; mp = mp->m_next);
				mp->m_next = omp;
			}
			break;

		case 'r':
			actionf = 1;
			while ((whfile = getfile()) != NIL) {
				struct mod			*omp;

				if ((mp = makelist(whfile)) == NIL) {
					fprintf(stderr, "%s : cannot open\n", whfile->ff_fnm);
					xerror();
				}
				/*
				 * remember that first node of "makelist"
				 * is actually file open (ACT_INF)
				 */
				if (*(mpp = find(&command, &mp->m_next->m_nam)) == NIL) {
					fprintf(stderr, "%s : cannot find module\n", radixout(&mp->m_next->m_nam));
					xerror();
				}
				omp = *mpp;
				*mpp = mp;
				mpp = &mp->m_next->m_next;
				mp = *mpp;
				*mpp = omp;
				omp->m_action &= ~ACT_PAS;
				while (mp->m_action != ACT_CLS) {
					if (*(mpp = find(mpp, &mp->m_nam)) == NIL) {
						fprintf(stderr, "%s : module not found\n", radixout(&mp->m_nam));
						xerror();
					}
					omp = *mpp;
					*mpp = mp;
					mpp = &mp->m_next;
					mp = *mpp;
					*mpp = omp;
					omp->m_action &= ~ACT_PAS;
				}
				mp->m_next = *mpp;
				*mpp = mp;
			}
			break;

		case 'x':
			extractf = 1;
			while ((whmod = getarg()) != NIL) {
				if ((mp = *find(&command, whmod)) == NIL) {
					fprintf(stderr, "%s : cannot find module\n", radixout(whmod));
					xerror();
				}
				mp->m_action |= ACT_XTR;
			}
			break;
		}
	}
}

/*
 * find module with "name" in list
 */
struct mod **
find(list, name)
struct mod			**list;
rad50n				*name;
{
	register struct mod		**mpp;
	register struct mod		*mp;

	for (mpp = list; (mp = *mpp) != NIL; mpp = &mp->m_next)
		if (mp->m_action&ACT_PAS && RADEQ(mp->m_nam, *name))
			break;
	return(mpp);
}

openi(ffp)
struct file			*ffp;
{
	register FILE			*fp;
	register struct fbuf		*fbp;
	register struct fbuf		**fbpp;

	if (ffp->ff_fb != NIL)
		return(0);
	if ((fp = fopen(ffp->ff_fnm, "r")) == NULL)
		return(-1);
	for (fbpp = &froot; (fbp = *fbpp) != NIL; fbpp = &fbp->ff_next)
		if (fbp->ff_io == NULL)
			break;
	if (fbp == NIL)
		*fbpp = fbp = malloc(sizeof(struct fbuf));
	ffp->ff_fb = fbp;
	fbp->ff_io = fp;
	setbuf(fp, fbp->ff_buf);
	return(0);
}

closei(ffp)
register struct file		*ffp;
{
	fclose(ffp->ff_fb->ff_io);
	ffp->ff_fb->ff_io = NULL;
	ffp->ff_fb = NIL;
}

struct mod *
makelist(ffp)
struct file			*ffp;
{
	register struct mod		*mp;
	int				t;
	struct mod			*mmp;
	FILE				*fp;
	int				ignore;
	struct glist			*ggp;

	if ((fp = fopen(ffp->ff_fnm, "r")) == NULL)
		return(NIL);
	setbuf(fp, obuf);

	ggp = NIL;
	mmp = mp = malloc(sizeof(struct mod));
	mp->m_action = ACT_INF;
	mp->m_file = ffp;
	while ((t = getrec(fp)) != 0)
		switch (t) {

		case GSD:
			while ((t = getgsd()) >= 0)
				switch (t) {

				case MDN:
					mp->m_next = malloc(sizeof(struct mod));
					mp = mp->m_next;
					mp->m_action = ACT_PAS;
					mp->m_file = ffp;
					mp->m_nam = gsd.gsd_nm;
					if (ggp != NIL) {
						ignore = 1;
						mp->m_dgbl = ggp;
						ggp = NIL;
					} else
						ignore = 0;
					break;

				case GSN:
					{
						register struct glist		*gp;

						if ((gsd.g_flags&DEF) && !ignore) {
							gp = malloc(sizeof(struct glist));
							gp->gl_nam = gsd.gsd_nm;
							gp->gl_nxt = mp->m_dgbl;
							mp->m_dgbl = gp;
						}
						if (!(gsd.g_flags & DEF)) {
							gp = malloc(sizeof(struct glist));
							gp->gl_nam = gsd.gsd_nm;
							gp->gl_nxt = mp->m_rgbl;
							mp->m_rgbl = gp;
						}
					}
					break;

				case PVI:
					mp->m_ver = gsd.gsd_nm;
					break;

				}

		case LMOD:
			for (t = 0; t < bcount; t += sizeof(rad50n)) {
				register struct glist		*gp;

				gp = malloc(sizeof(struct glist));
				gp->gl_nxt = ggp;
				gp->gl_nam = *((rad50n *)bpoint)++;
				ggp = gp;
			}
			break;

		}
	mp->m_next = malloc(sizeof(struct mod));
	mp = mp->m_next;
	mp->m_action = ACT_CLS;
	mp->m_file = ffp;
	fclose(fp);
	return(mmp);
}

char *
malloc(n)
{
	register char			*s;

	while (memlow+n >= memhigh) {
		if (sbrk(GOBBLE) == -1) {
			fprintf(stderr, "Core allocation failure\n");
			xerror();
		}
		memhigh += GOBBLE;
	}
	s = memlow;
	memlow += n;
	if (memlow > memmax)
		memmax = memlow;
	return(s);
}

free()
{
}

movb(fp, cp, ccc)
register FILE			*fp;
register char			*cp;
unsigned int			ccc;
{
	register unsigned int		cc;
	int				i;

	cc = ccc;
	do {
		if ((i = getc(fp)) == EOF)
			return(0);
		*cp++ = i;
	} while (--cc != 0);
	return(ccc);
}

getgsd()
{
	if (bcount < sizeof(struct gsd))
		return(-1);
	gsd = *((struct gsd *)bpoint)++;
	bcount -= sizeof(struct gsd);
	return(gsd.g_type);
}

/*
 * read paper tape style record
 */
getrec(fp)
register FILE			*fp;
{
	char				c;
	short				t;

	do {
		if (movb(fp, &c, 1) == 0)
			return(0);
	} while (c == 0);
	movb(fp, &c, 1);	/* skip 0 byte */
	movb(fp, &bcount, 2);
	movb(fp, &t, 2);
	if ((bcount -= 6) != 0)
		movb(fp, binbuf, bcount);
	movb(fp, &c, 1);	/* checksum */
	bpoint = binbuf;
	return(t);
}

/*
 * write paper tape style record
 */
putrec(fp, t)
FILE				*fp;
int				t;
{
	register unsigned int		cc;
	register char			*cp;
	register short			chksum;

	cc = bcount;
	cp = binbuf;
	putw(01, fp);
	putw(cc+6, fp);
	putw(t, fp);
	chksum = 1 + cc+6 + ((cc+6) >> 8) + t;
	while (cc-- > 0) {
		putc(*cp, fp);
		chksum += *cp++;
	}
	putc(-chksum & 0377, fp);
}

pass2()
{
	register struct mod		*mp;
	struct file			*ffp;
	register FILE			*fp;
	register struct glist		*gp;
	int				t;
	FILE				*xfile;
	char				*xbuf;
	char				xname[6+1+3+1];

	if (extractf)
		xbuf = malloc(BUFSIZ);
	for (mp = command; mp != NIL; mp = mp->m_next) {
		ffp = mp->m_file;
		if (ffp->ff_fb != NIL)
			fp = ffp->ff_fb->ff_io;
		switch (mp->m_action) {

		case ACT_INF:
			if (openi(ffp) < 0) {
				fprintf(stderr, "%s : cannot open?\n", ffp->ff_fnm);
				xerror();
			}
			break;

		case ACT_CLS:
			closei(ffp);
			break;

		case 0:
			while (getrec(fp) != EMOD);
			break;

		default:	/* ACT_PAS, ACT_XTR, ACT_PAS|ACT_XTR */
		case ACT_PAS:
			if (actionf) {
				int				gl_flag;

				gl_flag = 0;
				bcount = 0;
				bpoint = binbuf;
				for (gp = mp->m_dgbl; gp != NIL; gp = gp->gl_nxt) {
					*((rad50n *)bpoint)++ = gp->gl_nam;
					bcount += sizeof(rad50n);
					if (bcount == 32) {
						putrec(ofile, LMOD);
						gl_flag++;
						bcount = 0;
						bpoint = binbuf;
					}
				}
				if (bcount!=0 || gl_flag==0)
					putrec(ofile, LMOD);
			}
			if (mp->m_action & ACT_XTR) {
				strcpy(xname, radixout(&mp->m_nam));
				strcat(xname, ".obj");
				if ((xfile = fopen(xname, "w")) == NULL) {
					fprintf(stderr, "Can't create : %s\n", xname);
					xerror();
				}
				setbuf(xfile, xbuf);
			}
			do {
				t = getrec(fp);
				if (t != LMOD) {
					if (actionf)
						putrec(ofile, t);
					if (mp->m_action & ACT_XTR)
						putrec(xfile, t);
				}
			} while (t != EMOD);
			if (mp->m_action & ACT_XTR)
				fclose(xfile);
			break;
		}
	}
}

crefout()
{
	register struct mod		*mp;
	int				lcount;
	register struct glist		*gp;
	register struct ref		*rp;
	char				*smemlow;
	struct mod			*mp1;
	struct glist			*gp1;

	printf("\n\nCross Reference\n\nModule\tReferences\n\n");
	for (mp = command; mp != NIL; mp = mp->m_next)
		if (mp->m_action & ACT_PAS) {
			printf("%s", radixout(&mp->m_nam));
			lcount = 7;
			for (gp = mp->m_rgbl; gp != NIL; gp = gp->gl_nxt) {
				if (--lcount < 0) {
					lcount = 6;
					putchar('\n');
				}
				printf("\t%s", radixout(&gp->gl_nam));
			}
			putchar('\n');
		}

	printf("\n\n\nModule\tImplies\n\n");
	smemlow = memlow;

	for (mp1 = command; mp1 != NIL; mp1 = mp1->m_next)
		if (mp1->m_action & ACT_PAS) {
			struct ref			*undlist;
			struct ref			*implist;

			memlow = smemlow;
			implist = NIL;
			undlist = NIL;
			for (gp1 = mp1->m_rgbl; gp1 != NIL; gp1 = gp1->gl_nxt) {
				for (mp = mp1->m_next; mp != NIL; mp = mp->m_next)
					if (mp->m_action & ACT_PAS)
						for (gp = mp->m_dgbl; gp != NIL; gp = gp->gl_nxt)
							if (RADEQ(gp->gl_nam, gp1->gl_nam)) {
								/* do we already have it? */
								for (rp = implist; rp != NIL; rp = rp->r_next)
									if (RADEQ(*rp->r_name, mp->m_nam))
										goto gotit;
								rp = malloc(sizeof(struct ref));
								rp->r_name = &mp->m_nam;
								rp->r_next = implist;
								implist = rp;
								goto gotit;
							}
				rp = malloc(sizeof(struct ref));
				rp->r_name = &gp1->gl_nam;
				rp->r_next = undlist;
				undlist = rp;
			gotit:;
			}

			printf("%s", radixout(&mp1->m_nam));
			rlout(implist);
			if (undlist != NIL) {
				printf("\nundef:");
				rlout(undlist);
			}
			printf("\n\n");
		}

	memlow = smemlow;
	{
		register short			*p;

		for (p = memlow; p < memhigh; p++)
			*p = 0;
	}
}

/*
 * print out a ref. list
 */
rlout(rrp)
struct ref			*rrp;
{
	register struct ref		*rp;
	register int			lcount;

	lcount = 7;
	for (rp = rrp; rp != NIL; rp = rp->r_next) {
		if (--lcount < 0) {
			putchar('\n');
			lcount = 6;
		}
		printf("\t%s", radixout(rp->r_name));
	}
}
