/*
 *	lpd -- print files spooled by lpr
 *
 *	This is the main process communicating with the printers.
 *	As it releases the printer between jobs, other processes may grab it,
 *	but such activities should be minimised.
 */

#include <sgtty.h>
#include <signal.h>
#include <printers.h>
#include <fcntl.h>
#include <lp.h>
#include <types.h>
#include <dir.h>
#include <stat.h>
#include <stdio.h>
#include <errno.h>		/* DEBUG */

struct printer *prp;
struct lpdlock lpdlock;

FILE	*prfp;		/* file structure for lp */
short	lockfd;		/* file desc. of lock file */
short	elp;		/* cancel print flag */
FILE	*ifp;		/* pointer to print file */
short	dirfd;		/* file desc. of xdirp */
char	waiting;	/* set when in hibernation */
char	closedown;	/* if set, exit when current job finished */
char	push;		/* set if job is to be pushed out of printer */
#define	PUSH_SLEEP 5	/* seconds sleep before pushing */

char	lock[] =	"lock0";
#define	LPOSN	4
char	xdirp[] =	"0/";
#define	XPOSN	0
char	prname[] =	"/dev/lp\0";
#define	PPOSN	7
char	job[sizeof xdirp + 14];

#define NLINES 20	/* store this much overprinting - see "speed" */
char	*line[NLINES];
short	hiwater[NLINES];
short	lini;		/* line counter */
short	lino;		/* count of lines actually output */
short	column;
short	pwidth;
short	plength;	/* actual printing area on page (lines) */
short	lbanner;	/* line number offset caused by banner */

char	rstrtmsg[] =	"\n\n\n *** print restarted ***\n";
#define	RSTRTLINES	4	/* count lines in message */
char	cnclmsg[] =	"\n\n\n *** print cancelled ***\n";
#define	CNCLLINES	4	/* count lines in message */

extern errno;			/* DEBUG */
#define	MOTD "/etc/motd"

extern char *strcpy();
extern char *strcat();


main(ac, av)
char *av[];
{
	register z;
	int prstop();
	int restart();
	int getgoing();
	int clock();
	int shutdown();

	signal(SIGINT, SIG_IGN);
	signal(SIGQUIT, SIG_IGN);
	signal(SIGKICK, SIG_IGN);
	signal(SIGSTOP, prstop);
	signal(SIGRESTART, restart);
	signal(SIGALRM, clock);
	signal(SIGTERM, shutdown);

	if (ac == 2)		/* lpd for alternate printer */
	{
		if ((z = av[1][0]) == '-')
			z = av[1][1];	/* skip leading '-' */
		for (prp = printer; prp < &printer[NPRINTERS]; prp++)
			if (prp->ident == z)
				break;
		if (prp == &printer[NPRINTERS])
		{
			fprintf(stderr, "Illegal printer number\n");
			exit(1);
		}
		lock[LPOSN] = xdirp[XPOSN] = prname[PPOSN] = z;
	}
	else
	{
		fprintf(stderr, "No printer number specified\n");
		exit(1);
	}
	if (chdir(LPDDIR) == -1)
	{
		fprintf(stderr, "Can't chdir to %s\n", LPDDIR);
		exit(1);
	}
	if ((z = open(lock, 2)) != -1)
	{
		if (read(z, &lockfd, 2) == 2 && kill(lockfd, 0) != -1)
		{
			fprintf(stderr, "lpd already active\n");
			exit(1);
		}
		lseek(z, 0L, 0);
	}
	else
	{
		mknod(lock, S_IFALK | 0600, 0);
		z = open(lock, O_RDWR);
	}
	if (z == -1)
	{
		fprintf(stderr, "cannot create lock file\n");
		exit(1);
	}
	lockfd = z;

	if (fork())
		exit(0);

	writelock(lockfd);
	lpdlock.lpd_pid = getpid();
	lpdlock.lpd_uid = 0;
	lpdlock.lpd_pages = 0;
	write(lockfd, &lpdlock, sizeof lpdlock);
	unlock();

	for (;;)
	{
		/*
		 * The printers are single-open devices.
		 * Thus if the open fails, sleep and try again.
		 */
		lockwr(PR_OP);
		while (closedown == 0 &&
		    (z = open(prname, O_WRITE|O_EXCL)) == -1)
		{
if (errno == EINTR) printf("\n\n***** %s open interrupted *****\n\n", prname);	/* DEBUG */
			alarm(10);
			pause();
			alarm(0);
		}
		if (prp->baudrate != PARALLEL)	/* assume serial LA180 */
		{
			struct sgttyb sgttyb;

			gtty(z, &sgttyb);
			sgttyb.sg_ispeed = sgttyb.sg_ospeed = prp->baudrate;
			sgttyb.sg_flags = prp->modes;
			stty(z, &sgttyb);
		}
		prfp = fdopen(z, "w");
		if ((dirfd = open(xdirp, 0)) == -1)
		{
			fprintf(stderr, "lpd can't open %s\n", xdirp);
			unlink(lock);
			exit(1);
		}

		lockwr(PRINT);
		while (closedown == 0 && next())
			process();

		if (push)
			for (z = 0; z < prp->push; z++)
				fprintf(prfp, "\f");
		if (closedown)
		{
			unlink(lock);
			exit(0);
		}
/* DEBUG
		fclose(prfp);	/* free the printer */
if (fclose(prfp) == EOF && errno == EINTR)	/* DEBUG */
printf("\n\n***** %s close interrupted *****\n\n", prname);	/* DEBUG */
		close(dirfd);	/* in case directory is remade */
		waiting++;
		signal(SIGKICK, getgoing);
		lpdlock.lpd_uid = 0;
		lockwr(IDLE);
		while (waiting > 0)	/* only wakes after SIGKICK */
			pause();
	}
}

/*
 * next() finds the next lexicographically-ordered file
 * in the appropriate directory - it ignores
 * files starting with '.' and thus avoids
 * the . and .. directories.
 * Return is zero if nothing found, non-zero otherwise
 * (with file name set up in global array "job").
 */
next()
{
	register struct	direct *d1, *d2;
	struct direct dbuf[2];
	register i;

	lseek(dirfd, 0L, 0);
	d1 = &dbuf[0];
	d2 = &dbuf[1];
	d1->d_ino = 0;
	i = 0;
	while (read(dirfd, d1, sizeof(struct direct)) == sizeof(struct direct))
		if (d1->d_ino && d1->d_name[0] != '.')
		{
			i++;
			break;
		}
	if (i == 0)
		return(0);
	while (read(dirfd, d2, sizeof(struct direct)) == sizeof(struct direct))
	{
		if (d2->d_ino == 0 || d2->d_name[0] == '.')
			continue;
		if (strcmp(&(d1->d_name[2]), &(d2->d_name[2])) > 0)
		{
			i = (int)d1;
			d1 = d2;
			d2 = (struct direct *)i;
		}
	}
	strcat(strcpy(job, xdirp), d1->d_name);
	return(1);
}

process()
{
	register z, df;
	register char *buf;
	short ncopies;
	char mesg[LPDMSZ];
	char buff[512];		/* must be > LPDMSZ */

	lbanner = 0;
	push = 0;
	buf = buff;
	ncopies = 1;
	mesg[0] = 0;
	plength = prp->pagelength;
	if ((df = open(job, 0)) == -1)
		return;		/* go back and try another one */
	while (read(df, buf, LPDMSZ) == LPDMSZ)
	{
		buf[LPDMSZ] = 0;
		switch(buf[0])
		{
		case LP_UID:
			lpdlock.lpd_uid = (buf[2] << 8) | (buf[1] & 0377);
			lockwr(PRINT);
			elp = 0;	/* ignore previous signals */
			continue;

		case LP_BANNER:
			banner(&buf[1]);
			continue;

		case LP_PUSH:
			push++;		/* push job out of printer */
			continue;

		case LP_COPIES:
			ncopies = atoi(&buf[1]);
			if (ncopies <= 0)
				ncopies = 1;
			continue;

		case LP_NOSKIP:
			plength = prp->realpagelen;
			continue;

		case LP_FILE:
			{
				short	cntr;

				if ((ifp = fopen(&buf[1], "r")) == NULL)
					continue;
				cntr = ncopies;
				/*
				 * Should be top of page unless LA180 banner
				 * printed and even then don't want page skip
				 */
				lino = 0;
				do
				{
					if (lino)	/* force page top */
					{
						fprintf(prfp, "\f");
						lbanner = 0;
					}
					lino = lbanner;
					fseek(ifp, 0L, 0);
					speed();
					if (elp == 2)	/* restart */
					{
						elp = 0;
						cntr++;
						fprintf(prfp, rstrtmsg);
						lino += RSTRTLINES;
					}
					lino %= plength;
					lbanner = lino;
				} while (--cntr && elp == 0);
				fclose(ifp);
				if (elp)	/* prstop */
				{
					fprintf(prfp, cnclmsg);
					lino += CNCLLINES - plength;
					elp = 0;
				}
				if (lino)	/* leave idle at page top */
					fprintf(prfp, "\f");
				lbanner = 0;
				continue;
			}

		case LP_RM:
		case LP_RM_C:
			unlink(&buf[1]);
			continue;

		case LP_MESG:
			strcpy(mesg, &buf[1]);
			continue;

		case LP_TTY:
			{
				FILE *tfp;

				tfp = fopen(&buf[1], "w");
				fprintf(tfp, "\007\n%s: job printed\n",
					*mesg ? mesg : "");
				fclose(tfp);
			}
			continue;

		case LP_PAGES:
			{
				struct lpdlock	lpdlock;

				writelock(lockfd);
				lseek(lockfd, 0L, 0);
				read(lockfd, &lpdlock, sizeof lpdlock);
				lpdlock.lpd_pages -= (buf[2] << 8) | (buf[1] & 0377);
				lseek(lockfd, 0L, 0);
				write(lockfd, &lpdlock, sizeof lpdlock);
				unlock();
			}
			continue;

		default:
			fprintf(stderr, "lpd: corrupt control file\n");
			link(job, "corrupt");
			break;
		}
		break;
	}
	close(df);
	unlink(job);
	fflush(prfp);
	if (closedown == 0 && next() == 0)
	{
		lpdlock.lpd_uid = 0;
		lockwr(IDLE);
		signal(SIGKICK, getgoing);
		alarm(PUSH_SLEEP);
		pause();
		alarm(0);
		signal(SIGKICK, SIG_IGN);
	}
}

lockwr(state)
{
	struct lpdlock	oldlpdlock;

	writelock(lockfd);
	lseek(lockfd, 0L, 0);
	read(lockfd, &oldlpdlock, sizeof oldlpdlock);
	lpdlock.lpd_pages = oldlpdlock.lpd_pages;
	lpdlock.lpd_state = state;
	lseek(lockfd, 0L, 0);
	write(lockfd, &lpdlock, sizeof lpdlock);
	unlock();
}


banner(s)
char *s;
{
	register n;
	long timbuf;
	extern long time();

	time(&timbuf);
	if (prp->baudrate == PARALLEL)
	{
		register char *i, *j;
		char fancyname[42];

		rule();
		fprintf(prfp, "\n\n\n\n\n\t\tunix edition-7 local print\n\n");
		fprintf(prfp, "\t\t\t\t\t\t%s\n\n", ctime(timbuf));
		i = fancyname;
		for (n = 0; n < 12; n++)
			*i++ = ' ';
		while (n++ < 17)
			*i++ = '*';
		*i++ = ' ';
		n++;
		j = s;
		while (n++ < 38)
			if ((*i++ = *j++) == 0)
				break;
		i--;
		*i++ = '\n'; *i++ = '\n'; *i++ = 0;
		for (n = 0; n < 12; n++)
		{
			fprintf(prfp, "\t\t\t\t\t");
			fwrite(&fancyname[11 - n], sizeof(char), i - fancyname - 12 + n, prfp);
		}
		note();
		rule();
	}
	else
	{
		fprintf(prfp, "%s\t\t%s", s, ctime(timbuf));
		n = ((strlen(s) + 16) & ~07) + 24;
		while (n--)
			fprintf(prfp, "-");
		fprintf(prfp, "\n");
		lbanner += 2;
	}
}

rule(size)	/* skip to perforations, print a line, and then form-feed */
{
	register i;
	register char *cp;
	char perfs[150];	/* no printer here is wider than this! */

	cp = perfs;
	*cp++ = PERFSKP;
	for (i = 0; i < prp->pagewidth; i++)
		*cp++ = '0';
	*cp++ = '\r';
	fwrite(perfs, sizeof(char), cp - perfs, prfp);
	*cp++ = '\f';
	fwrite(&perfs[1], sizeof(char), cp - perfs - 1, prfp);
}

note()
{
	register fd, i;
	register char *nbuf;
	char notebuf[512];

	nbuf = notebuf;
	if ((fd = open(MOTD, 0)) == -1)
		return;
	fprintf(prfp, "\n\n\t\t\t\tUnix Note\n\n\n");
	while ((i = read(fd, nbuf, sizeof notebuf)) > 0)
		fwrite(nbuf, sizeof(*nbuf), i, prfp);
	close(fd);
}

prstop()
{
	struct lpiocb lpiocb;

	signal(SIGSTOP, prstop);
	elp = 1;
	fseek(ifp, 0L, 2);	/* Cause end of file for the print file */
	if (prp->baudrate == PARALLEL)
	{
		lpiocb.lp_flag = FLSH;
		ioctl(fileno(prfp), LPSET, &lpiocb);
		flushline('\n');
		fflush(prfp);
		while (ioctl(fileno(prfp), LPGET, &lpiocb) == 0 &&
		    (lpiocb.lp_flag & FLSH))
		{
			alarm(2);
			pause();
			alarm(0);
		}
	}
}

restart()
{
	signal(SIGRESTART, restart);
	elp = 2;
	fseek(ifp, 0L, 2);	/* Cause end of file for the print file */
}

getgoing()
{
	signal(SIGKICK, SIG_IGN);
	waiting = 0;
}

shutdown()
{
	signal(SIGTERM, SIG_IGN);
	closedown++;
	lockwr(CLDN);
	getgoing();
}

clock()
{
	signal(SIGALRM, clock);
}

/*
 * "speed" attempts to optimise overstriking, and reduce paper wastage.
 * It stores up to NLINES overstruct characters per column position
 * before flushing the line out. It absorbs multiple formfeeds and linefeeds
 * at the top of a page.
 */

speed()
{
	register short c, trns, plim;

	lini = lino;
	column = 0;
	trns = prp->translate;
	pwidth = prp->pagewidth;
	for (;;)
	{
		switch(c = getc(ifp))
		{
		case EOF:
			if (hiwater[0])		/* no \n at end */
				flushline('\n');	/* so supply one */
			return;

		case '\n':
			if (lino || hiwater[0])	/* something on this page */
			{
				lino++;
				flushline('\n');
			}
			lini++;
			column = 0;
			if (lini < plength || plength == prp->realpagelen)
				continue;
		case '\f':
			lini = 0;
			if (lino == 0 && hiwater[0] == 0)
				continue;	/* absorb repeated form-feeds */
			flushline('\f');
			column = 0;
			lino = 0;
			continue;

		case ' ':
			column++;
			continue;

		case '\t':
			column = (column + 8) & ~07;
			continue;

		case '\r':
			column = 0;
			continue;

		case '\b':
			if (--column < 0)
				column = 0;
			continue;

		case '{':
			if (trns)
			{
				savch('-');
				c = '(';
			}
			break;

		case '}':
			if (trns)
			{
				savch('-');
				c = ')';
			}
			break;

		case '~':
			if (trns)
			{
				savch('-');
				c = '^';
			}
			break;

		case '|':
			if (trns)
			{
				savch('-');
				c = '!';
			}
			break;

		case '`':
			if (trns)
			{
				savch('-');
				c = '\'';
			}
			break;

		case PERFSKP:
			lini = 0;
			flushline(PERFSKP);
			column = 0;
			lino = 0;
			continue;

		case NAUTEJ:
			plength = prp->realpagelen;
			continue;

		case AUTEJ:
			plength = prp->pagelength;
			continue;

		default:
			if ((c < ' ') || (c >= 0177)) 	/* unprintable -- ignore */
				continue;
		}
		savch(c);
		column++;
	}
}

savch(c)
char c;
{
	register i;
	register char *p, *q;
	char *malloc();

	if (lini != lino)
	{
		lini = (lini - lino) % plength;
		for (i = 0; i < lini; i++)
			flushline('\n');
		lino = lini;
	}
	if (column < pwidth)
	{
		for (i = 0; i < NLINES; i++)
			if (p = line[i])
			{
				if (p[column] == 0)
				{
					p[column] = c;
					if (column >= hiwater[i])
						hiwater[i] = column + 1;
					break;
				}
			}
			else
			{
				p = malloc(pwidth);
				if (p == NULL)
				{
					fprintf(stderr, "Out of core\n");
					exit(1);
				}
				line[i] = p;
				for (q = p; q < &p[pwidth]; *q++ = 0);
				p[column] = c;
				hiwater[i] = column + 1;
				break;
			}
		if (i == NLINES)
		{
			flushline('\r');
			line[0][column] = c;
			hiwater[0] = column + 1;
		}
	}
}

flushline(c)
{
	register i, j;

	for (i = 0; hiwater[i] && i < NLINES; i++)
	{
		for (j = 0; j < hiwater[i]; j++)
		{
			if (line[i][j])
			{
				putc(line[i][j], prfp);
				line[i][j] = 0;
			}
			else
				putc(' ', prfp);
		}
		hiwater[i] = 0;
		if (i < NLINES - 1 && hiwater[i + 1])
			putc('\r', prfp);
	}
	putc(c, prfp);
}
