/*
**	netget [-flags] [filename]
**
**	Get a file queued from the network
**
**		default		get first user's file found
**		-L		retrieve lost files
**		-h<host>	restrict to files from <host>
**		-k		keep file in files area
**		-l		put output on stdout
**		-o<name>	renames output to <name>
**		-t		just report files spooled
**		-u<user>	restrict to files from <user>
**		-v		print statistics and announce the presence of other files
**		-y		assume "yes" to questions
**		filename	get specific file if it exists
**
**	SETUID -> ROOT
**
**	Bugs and comments to:	Piers Lauder
**				Dept of Comp Sci
**				Sydney University
**	August '80.
*/

#include	<local-system>
#ifdef	Aus2
#include	<types.h>
#include	<stat.h>
#include	<dir.h>
#endif	Aus2
#ifdef	Aus1
#include	<dir.h>
#include	<stat16.h>
#include	"lvl7.h"
#endif	Aus1
#include	<errno.h>
#include	<passwd.h>
#include	<stdio.h>
#include	<ctype.h>
#include	"net.h"
#include	"neth.h"


char		filesdir[3*(DIRSIZ+1)];
FILE *		filesfd;
char *		name;
char *		file;
char 		username[MAXUNAME];
char *		rhost;
char *		ruser;
char *		rename;
char		memerr[]		= "out of memory";
short		Yes;
short		spooled;
short		Verbose;
short		Keep;
short		Report;
short		Stdout;
short		delete;
short		lostflag;
short		Umask;
short		exists;
struct pwent	pe;

short		getmode();
time_t		getdate();

extern char *	strncpy();
extern char *	strcpy();
extern char *	strchr();
extern char *	strrchr();
extern char *	strncat();
extern char *	strcat();
extern char *	malloc();
extern char *	realloc();
extern char	end;

#define	STREQUAL	0
#define	NULLSTR		(char *)0



main(argc, argv)
	register int	argc;
	register char *	argv[];
{
	register char	c;
	char		buf[SSIZ];
	char	outbuf[BUFSIZ];

	name = *argv;

	strcat(strcpy(filesdir, SPOOLDIR), FILESDIR);
	if ( (filesfd = fopen(filesdir, "r")) == NULL )
		neterror("cannnot read %s", filesdir);

	pe.pw_uid = getuid();
	if ( getpwlog(&pe, buf, BUFSIZ) == PWERROR )
		neterror("whose is uid %d?", pe.pw_uid);

	pwclose();

	while ( --argc )
	{
		if ( **++argv == '-' )
		{
			while ( c = *++*argv )
			{
				switch ( c )
				{
				 case 'L':	if ( pe.pw_uid )	continue;
						pe.pw_strings[LNAME] = "lost";
						lostflag++;
						continue;
				 case 'h':	rhost = ++*argv;	break;
				 case 'k':	Keep++;			continue;
				 case 'l':	Stdout++;		continue;
				 case 'o':	rename = ++*argv;	break;
				 case 't':	Report++;		continue;
				 case 'n':	/* alt. for 'u' */
				 case 'u':	ruser = ++*argv;	break;
				 case 'v':	Verbose++;		continue;
				 case 'y':	Yes++;			continue;
				 default:	neterror("unrecognised flag '%c'", c);
						continue;
				}
				break;
			}
		}
		else
			if ( file != NULLSTR )
				neterror("only one file arg allowed");
			else
				file = *argv;
	}

	strncpy(username, pe.pw_strings[LNAME], MAXUNAME);
	if ( (c = strlen(username)) < MAXUNAME )
		do username[c] = '.'; while ( ++c < MAXUNAME );

#	ifdef	Aus2
	Umask = umask(0777);
#	endif

	if ( Stdout )
		setbuf(stdout, outbuf);

	if ( file )
	{
		if ( findfile(0) )
			movefile();
		else
			neterror("%s not found", file);
	}
	else
	{
		if ( rename )
			neterror("exactly which file do you want renamed?");
		while ( findfile(1) )
			movefile();
	}

	if ( !spooled )
	{
		fprintf(stderr, "No files spooled.\n");
		return 1;
	}

	return 0;
}



/*
**	Search filesdir for correctly named files,
**	and read header for file name.
**
**	If interacting, prompt user with file name,
**	and create it in current directory,
**	otherwise look for match with "file".
*/

findfile(interact)
	int		interact;
{
	register	found = 0;
	register FILE *	filedesc;
	long		headsize;
	static char	filenmb[sizeof filesdir + DIRSIZ];
	struct direct	dir;

	while ( fread((char *)&dir, sizeof dir, 1, filesfd) == 1 )
	{
		if ( dir.d_ino == 0 || dir.d_name[0] == '.' )
			continue;
		if ( strncmp(dir.d_name, username, MAXUNAME) == STREQUAL )
		{
			strncat(strcpy(filenmb, filesdir), dir.d_name, DIRSIZ);
			filenmb[sizeof filenmb - 1] = '\0';
			if ( (filedesc = fopen(filenmb, "r")) == NULL )
				neterror("cannot read %s", filenmb);
			headsize = netheader(filedesc);
			fclose(filedesc);
			if ( strcmp(pe.pw_strings[LNAME], destname) != STREQUAL && !lostflag )
				continue;

			spooled++;

			if ( rhost && strcmp(rhost, orghost) != STREQUAL
				|| ruser && strcmp(ruser, orgname) != STREQUAL
			   )
				continue;

			if ( Report )
			{
				time_t	t = getdate();

				fprintf(stdout, "\"0%o %15.15s %s\" from %s at %s\n"
						,getmode()
						,ctime(t)+4
						,datname
						,orgname
						,orghost
					);
				if ( Verbose )
					showstats(filenmb, headsize);
				continue;
			}

			if ( !interact && strcmp(file, datname) == STREQUAL )
			{
				found++;
				break;
			}
			if ( interact && validate() )
			{
				found++;
				break;
			}
		}
	}

	if ( Report || spooled == 0 )
		return 0;

	if ( spooled > 1 && Verbose )
		fprintf(stdout, "%d other file%s spooled for you\n", spooled-1, spooled>2?"s":"");

	if ( !found )
	{
		if ( !interact )
			fprintf(stderr, "%s not found.\n", file);
		return 0;
	}

	if ( Verbose )
		showstats(filenmb, headsize);

	file = filenmb;
	return 1;
}



/*VARARGS1*/
neterror(s, a)
	char *	s;
	char *	a;
{
	extern	errno;

	fprintf(stderr, "%s: ", name);
	fprintf(stderr, s, a);
	if ( errno )
		perror("\07");
	else
		fprintf(stderr, "\nUsage: \"%s [-flags ...] [filename]\"\n", name);
	exit(1);
}



movefile()
{
	register	c;
	register FILE *	ofd;
	register	to_tty = 0;
	register FILE *	filedesc;
	char *		save;
	extern		errno;

	if ( delete )
	{
		delete = 0;
		goto out;
	}

	if ( Stdout )
	{
		if ( isatty(fileno(stdout)) )
			to_tty = 1;
		ofd = stdout;
	}
	else
	{
		if ( rename )
		{
			tfree(datname);
			datname = rename;
			rename = NULLSTR;
		}

		if ( access(datname, 2) == SYSERROR )
		{
			if ( errno == ENOENT )
			{
				register char *	dir;

				if ( (dir = strrchr(datname, '/')) == NULLSTR )
					dir = ".";
				else
				{
					static char *	temp;

					c = dir - datname;
					if ( temp != NULLSTR )
						temp = realloc(temp, c+1);
					else
						temp = malloc(c+1);
					if ( temp == NULLSTR )
						neterror(memerr);
					dir = temp;
					strncpy(dir, datname, c);
					dir[c] = '\0';
				}
				if ( access(dir, 2) == SYSERROR )
					neterror(datname);
			}
			else
				neterror(datname);

			exists = 0;
		}
		else
			exists = 1;

		if ( (ofd = fopen(datname, "w")) == NULL )
			neterror("cannot open %s", datname);

		if ( !exists )
			chown(datname, pe.pw_uid, pe.pw_gid);
	}

	filedesc = fopen(file, "r");
	save = datname;
	datname = 0;
	netheader(filedesc);
	tfree(datname);
	datname = save;

	while ( (c = getc(filedesc)) != EOF )
	{
		if ( !to_tty || isprint(c) || isspace(c) )
			c = putc(c, ofd);
		else
			c = putc('?', ofd);
		if ( c == EOF && ferror(ofd) )
			neterror("cannot write %s", datname);
	}

	if ( ferror(filedesc) )
		neterror("cannot read %s", file);

	if ( !Stdout )
	{
		fclose(ofd);
		if ( !exists )
#		ifdef	Aus1
			chmod(datname, getmode());
		smdate(datname, getdate());
#		endif
#		ifdef	Aus2
			chmod(datname, getmode()&~Umask);
		{
			struct utimbuf { time_t actime, modtime; } utb;

			utb.actime = utb.modtime = getdate();
			utime(datname, &utb);
		}
#		endif	Aus2
	}

	fclose(filedesc);

out:
	if ( !Keep && unlink(file) == SYSERROR )
		neterror("cannot unlink %s", file);
}



/*
**	Extract any date from actparam
*/

time_t
getdate()
{
	register char *	cp;
	extern long	atol();

	for ( cp = actparam ; (cp = strchr(cp, 'D')) != NULLSTR ; cp++ )
		if ( strncmp(cp, "DATE=", 5) == STREQUAL )
			return (time_t)atol(cp+5);

	if ( (cp = strchr(netstats, '@')) != NULLSTR )
		return (time_t)atol(cp+1);

#	ifdef	GMT
	return (time_t)(time((long *)0)+(long)GMT*60L*60L);
#	else
	return (time_t)time((long *)0);
#	endif
}



/*
**	Extract any mode from actparam
*/

short
getmode()
{
	register char *	cp;

	for ( cp = actparam ; (cp = strchr(cp, 'M')) != NULLSTR ; cp++ )
		if ( strncmp(cp, "MODE=0", 6) == STREQUAL )
		{
			register char	c;
			register	o;

			for ( o = 0, cp += 6 ; (c = *cp++) >= '0' && c <= '7' ; o = (o<<3)+(c-'0') );

#			ifdef	Aus2
			return o & 0777;
#			endif	Aus2
#			ifdef	Aus1
			return o & 0707;
#			endif	Aus1
		}

	return 0600;	/* default */
}



validate()
{
	register	ok;
	register	r;
	register char *	s;
	register	c;
	char *		temp;
	time_t		t;

	if ( Yes )
		return 1;

ask:
	if ( !Stdout && access(datname, 0) != SYSERROR )
	{
		s = " (already exists!) ";
		exists = 1;
	}
	else
	{
		s = " ";
		exists = 0;
	}

	if ( Stdout )
		fflush(stdout);

	t = getdate();
	fprintf(stderr, "\"%15.15s %s\" from %s at %s%s? ", ctime(t)+4, datname, orgname, orghost, s);

	r = 0;
	ok = 0;

	switch ( getc(stdin) )
	{
	 case 'd':	delete++;
	 case 'y':	ok++;	break;
	 case 'r':	r++;	break;
	 case '\n':	return 0;
	 case EOF:	exit(0);
	}

	while ( (c = getc(stdin)) != '\n' )
		if ( c == EOF )
			exit(0);

	if ( !r )
		return ok;

	if ( (temp = malloc(100)) == NULLSTR )
		neterror(memerr);

	for(;;)
	{
		fprintf(stderr, "rename: ");

		for ( s = temp, ok = 0 ; ok < 100 ; )
			switch ( r = getc(stdin) )
			{
			 case EOF:	exit(0);
			 case '\n':	*s++ = '\0';
					tfree(datname);
					datname = realloc(temp, s-temp);
					goto ask;
			 case '\t':
			 case ' ':	if ( !ok ) continue;
			 default:	*s++ = r; ok++;
			}

		fprintf(stderr, "rename too long!\n");
		while ( (c = getc(stdin)) != '\n' )
			if ( c == EOF )
				exit(0);
	}
}



showstats(filenmb, headsize)
	char *	filenmb;
	long	headsize;
{
#	ifdef	Aus2
	struct stat	statbuf;
#	define	stsize(A)	(A)->st_size
#	endif	Aus2
#	ifdef	Aus1
	struct statbuf	statbuf;
#	define	stsize(A)	(((long)((A)->sb_size0)<<16)|(A)->sb_size1)
#	endif	Aus1

	stat(filenmb, &statbuf);
	getstats(netstats, (time_t)0);
	setrates(stsize(&statbuf)-headsize);
	printstats(stdout);
}



tfree(buf)
	char *		buf;
{
	extern char	end;

	if ( buf > &end && buf < (char *)&buf )
		free(buf);
}
