/*
 *
 *	UNIX debugger
 *
 */

#include "defs.h"


char		LONGFIL[];
char		NOTOPEN[];
char		A68BAD[];
char		A68LNK[];
char		BADMOD[];

struct map	txtmap;
struct map	datmap;

struct symtab	symbol;
int		lastframe;
int		callpc;

int		infile;
int		outfile;
char		*lp;
int		maxoff;
int		maxpos;
int		octal;

/*
 * symbol management
 */
long int	localval;

/*
 * breakpoints
 */
struct bkpt	*bkpthead;

struct reglist reglist[] =
{
	"ps", ps,
	"pc", pc,
	"sp", sp,
	"r5", r5,
	"r4", r4,
	"r3", r3,
	"r2", r2,
	"r1", r1,
	"r0", r0,
};

char		lastc;
unsigned	corhdr[];
unsigned	*endhdr;

int		fcor;
char		*errflg;
int		signo;


long int	dot;
long int	var[];
char		*symfil;
char		*corfil;
int		pid;
long int	adrval;
int		adrflg;
long int	cntval;
int		cntflg;

char		*signals[] =
{
	"",
	"hangup",
	"interrupt",
	"quit",
	"illegal instruction",
	"trace/BPT",
	"IOT",
	"EMT",
	"floating exception",
	"killed",
	"bus error",
	"memory fault",
	"bad system call",
	"broken pipe",
	"alarm call",
	"terminated",
};


/*
 * general printing routines ($)
 */
printtrace(modif)
{
	int narg, i, stat, name, limit;
	unsigned dynam;
	register struct bkpt *bkptr;
	char hi, lo;
	int word;
	char *comptr;
	long int argp, frame, link;
	struct symtab *symp;

	if (cntflg == 0)
		cntval = -1;
	switch (modif)
	{
    case '<':
    case '>':
		{
			char file[64];
			int index;

			index = 0;
			if (modif == '<')
				iclose();
			else
				oclose();
			if (rdc() != EOR)
			{
				do
				{
					file[index++] = lastc;
					if (index >= 63)
						error(LONGFIL);
				} while (readchar() != EOR);
				file[index] = 0;
				if (modif == '<')
				{
					infile = open(file, 0);
					if (infile < 0)
					{
						infile = 0;
						error(NOTOPEN);
					}
				}
				else
				{
					outfile = open(file, 1);
					if (outfile < 0)
						outfile = creat(file, 0644);
					else
						lseek(outfile, 0L, 2);
				}
			}
			lp--;
		}
		break;

    case 'o':
		octal = TRUE;
		break;

    case 'd':
		octal = FALSE;
		break;

    case 'q':
    case 'Q':
    case '%':
		done();

    case 'w':
    case 'W':
		maxpos = (adrflg ? adrval : MAXPOS);
		break;

    case 's':
    case 'S':
		maxoff = (adrflg ? adrval : MAXOFF);
		break;

    case 'v':
    case 'V':
		prints("variables\n");
		for (i = 0; i <= 35; i++)
		{
			if (var[i])
			{
				printc((i <= 9 ? '0' : 'a' - 10) + i);
				printf(" = %Q\n", var[i]);
			}
		}
		break;

    case 'm':
    case 'M':
		printmap("? map", &txtmap);
		printmap("/ map", &datmap);
		break;

    case 0:
    case '?':
		if (pid)
			printf("pcs id = %d\n", pid);
		else
			prints("no process\n");
		sigprint();
		flushbuf();

    case 'r':
    case 'R':
		printregs();
		return;

    case 'f':
    case 'F':
		printfregs(modif == 'F');
		return;

    case 'c':
    case 'C':
		frame = (adrflg ? adrval : endhdr[r5]) & EVEN;
		lastframe = 0;
		callpc = (adrflg ? get(frame+2,DSP) : endhdr[pc]);
		while (cntval--)
		{
			chkerr();
			narg = findroutine(frame);
			printf("%.8s(", symbol.symc);
			argp = frame + 4;
			if (--narg >= 0)
				printf("%o", get(argp, DSP));
			while (--narg >= 0)
			{
				argp += 2;
				printf(",%o", get(argp, DSP));
			}
			prints(")\n");
			if (modif == 'C')
			{
				while (localsym(frame))
				{
					word = get(localval, DSP);
					printf("%8t%.8s:%10t", symbol.symc);
					if (errflg)
					{
						prints("?\n");
						errflg = 0;
					}
					else
						printf("%o\n",word);
				}
			}
			lastframe = frame;
			frame = get(frame, DSP) & EVEN;
			if (frame == 0)
				break;
		}
		break;

    case 'e':		/* print externals */
    case 'E':
		symset();
		while (symp = symget())
		{
			chkerr();
			if ((symp->symf) == 043 || (symp->symf) == 044)
				printf("%.8s:%12t%o\n",
					symp->symc, get(leng(symp->symv), DSP));
		}
		break;

    case 'a':
    case 'A':
		frame = (adrflg ? adrval : endhdr[r4]);
		while (cntval--)
		{
			chkerr();
			stat = get(frame, DSP);
			dynam = get(frame+2, DSP);
			link = get(frame+4, DSP);
			if (modif == 'A')
				printf("%8O:%8t%-8o,%-8o,%-8o",
					frame, stat, dynam, link);
			if (stat == 1)
				break;
			if (errflg)
				error(A68BAD);
			if (get(link-4, ISP) != 04767)
			{
				if (get(link-2, ISP) != 04775)
					error(A68LNK);
				else	/*compute entry point of routine*/
					prints(" ? ");
			}
			else
			{
				printf("%8t");
			        valpr(name = shorten(link) + get(link-2, ISP), ISYM);
				name = get(leng(name - 2), ISP);
				printf("%8t\"");
				limit = 8;
				do
				{
					word = get(leng(name), DSP);
					name += 2;
					lo = word & LOBYTE;
					hi = (word >> 8) & LOBYTE;
					printc(lo);
					printc(hi);
				} while (lo && hi && limit--);
				printc('"');
			}
			limit = 4;
			i = 6;
			printf("%24targs:%8t");
			while (limit--)
			{
				printf("%8t%o", get(frame+i, DSP));
				i += 2;
			}
			printc(EOR);
			frame = dynam;
		}
		errflg = 0;
		flushbuf();
		break;

    case 'b':		/* print breakpoints */
    case 'B':
		printf("breakpoints\ncount%8tbkpt%24tcommand\n");
		for (bkptr = bkpthead; bkptr; bkptr = bkptr->nxtbkpt)
		{
			if (bkptr->flag)
			{
				printf("%-8.8d", bkptr->count);
				psymoff(leng(bkptr->loc), ISYM, "%24t");
				comptr = bkptr->comm;
				while (*comptr)
					printc(*comptr++);
			}
		}
		break;

    default:
		error(BADMOD);
	}
}

printmap(s, amap)
char *s;
struct map *amap;
{
	int file;

	file = amap->ufd;
	printf("%s%12t`%s'\n", s,
		(file < 0 ? "-" : (file == fcor ? corfil : symfil)));
	printf("b1 = %-16Q", amap->b1);
	printf("e1 = %-16Q", amap->e1);
	printf("f1 = %-16Q", amap->f1);
	printf("\nb2 = %-16Q", amap->b2);
	printf("e2 = %-16Q", amap->e2);
	printf("f2 = %-16Q", amap->f2);
	printc(EOR);
}

printfregs(longpr)
{
	register i;
	double f;
	union knuckle
	{
		float f_fred;
		double f_nurk;
	};

	printf("fpsr	%o\n", corhdr[0].u_fps.u_fpsr);
	for (i = 0; i < FRMAX; i++)
	{
		if (corhdr[0].u_fps.u_fpsr & FD || longpr)	/* long mode */
			f = corhdr[0].u_fps.u_fpregs[i].f_nurk;
		else
			f = corhdr[0].u_fps.u_fpregs[i].f_fred;
		printf("fr%-8d%-32.18f\n", i, f);
	}
}

printregs()
{
	register struct reglist *p;
	int v;

	for (p = reglist; p < &reglist[9]; p++)
	{
		printf("%s%8t%o%8t", p->rname, v = endhdr[p->roffs]);
		valpr(v, (p->roffs == pc ? ISYM : DSYM));
		printc(EOR);
	}
	printpc();
}

getreg(regnam)
{
	register struct reglist *p;
	register char *regptr;
	char regnxt;

	regnxt = readchar();
	for (p = reglist; p < &reglist[9]; p++)
	{
		regptr = p->rname;
		if ((regnam == *regptr++) && (regnxt == *regptr))
			return(p->roffs);
	}
	lp--;
	return(0);
}

printpc()
{
	dot = endhdr[pc];
	psymoff(dot, ISYM, ":%16t");
	printins(0, ISP, chkget(dot, ISP));
	printc(EOR);
}

sigprint()
{
	prints(signals[signo]);
}
