/*
 *	disk copier and compacter
 *	J. N. Rottman
 *
 *	NOTE:
 *	1) Boostrap block is copied;
 *	2) Size of destination file system is prompted for - thus dcopy
 *	   should not be fired off asynchronously (but will work if it is);
 *	3) Disk errors are also copied, so dcheck and icheck before copying;
 *	4) Interleaves files according to magic numbers below;
 *	5) Compresses directories (removes empty slots).
 *
 *	C. H. Maltby
 *	K. F. Hill
 */

#include <param.h>
#include <filsys.h>
#include <fblk.h>
#include <ino.h>
#include <inode.h>
#include <dir.h>
#include <signal.h>
#include <stdio.h>

/*
 * number of directory entries per block
 */
#define	DIRPB	(BSIZE/sizeof(struct direct))

/*
 * some magic numbers
 */
#define	IN	7	/* magic interleave number */
#define NSEC	22	/* number of sectors per track */
#define	NTRACK	19	/* number of tracks per cylinder */

/*
 * NPOOL is the size of the output buffer pool,
 * and may be reduced if this program is too big
 * (dcopy brk's for one word for every inode on the source disk)
 */
#define	NPOOL	50


struct	filsys	super0;		/* old superblock */
struct	filsys	super1;		/* new superblock */
struct	dinode	node0;		/* current inode is copied to here */
struct	dinode	node1;		/* new inode is built here */

struct	dinode	in_node[INOPB];		/* input inode buffer */
struct	dinode	out_node[INOPB];	/* output inode buffer */

short	inum	= INOPB;	/* where up to in in_node */
daddr_t	iblk	= 1;		/* input inode block just read */
short	onum	= 0;		/* where up to in out_node */
daddr_t	oblk	= 2;		/* output inode block to be written */
short	*iindex;		/* addresses the in-core inode array */
int	fi;			/* input file descriptor */
int	fo;			/* output file descriptor */
int	fx;			/* file descriptor for geti */
daddr_t	lastblock;		/* last block written out */
short	inext	= 1;		/* inode numbers on new filsys */
short	dnum;			/* where up to in new directory data block */
short	dlevel;			/* level of indirection for new directory */
struct direct *dirblk;		/* new directory built here */
long	dirsiz;			/* size of directory being made */
daddr_t	bno[DNADDR];		/* expanded addresses of inode */
char	in[BSIZE];		/* file data buffer */
char	*argv0;			/* overwrite argv[0] with blk number */

struct	dbuf
{
	char	d_nleft;
	daddr_t	d_nr;
	daddr_t	*d_upto;
	daddr_t	*d_base;
} dbuf[4];			/* direct + 3 levels of indirection */

struct
{
	daddr_t	bn;		/* output block number */
	char	bb[BSIZE];	/* data */
} bpl[NPOOL];

int	inpool;			/* where up to in inpool */
int	maxpool;		/* upper bound on pool */

char	adr[NSEC], flag[NSEC];	/* for the interleaving template */


main(argc, argv)
char **argv;
{
	register int i, icount;
	register short *indx;
	daddr_t countfree();
	long getnum();
	extern char *brk(), *sbrk();

	if (argc != 3)
	{
		printf("Usage: dcopy fromfilesys tofilesys\n");
		exit(1);
	}
	if ((fi = open(argv[1], 0)) < 0)
	{
		printf("Cannot open %s\n", argv[1]);
		exit(1);
	}
	if ((fo = open(argv[2], 2)) < 0)
	{
		printf("Cannot open %s\n", argv[2]);
		exit(1);
	}
	argv0 = argv[0];
	maxpool = NPOOL;
	dbuf[0].d_base = &bno[0];
	dbuf[1].d_base = (daddr_t *)bpl[NPOOL-1].bb;
	dbuf[2].d_base = (daddr_t *)bpl[NPOOL-2].bb;
	dbuf[3].d_base = (daddr_t *)bpl[NPOOL-3].bb;
	sync();
	getblk(fi, (char *)&super0, (daddr_t)0);
	putblk((int *)&super0, (daddr_t)0);	/* copy bootstrap */
	getblk(fi, (char *)&super0, SUPERB);

	if (isatty(0))
	{
		daddr_t usedb, freeb;

		i = counti();			/* blks needed for inodes */
		freeb = countfree();		/* blocks in freelist */
		usedb = super0.s_fsize - (daddr_t)super0.s_isize - freeb;
		printf("%s: size = %ld blks (%ld in use); i-list = %d blks (%d in use)\n",
			argv[1], super0.s_fsize,
			super0.s_fsize - freeb,
			super0.s_isize - 2, i);
		i += 2;				/* min. new isize */
		for (;;)
		{
			printf("Enter size (blks) for %s file system ", argv[2]);
			super1.s_fsize = getnum(super0.s_fsize);
			if (super1.s_fsize < (long)i + usedb)
			{
				printf("Too small\n");
				continue;
			}
			printf("Enter size (blks) of i-list (%d inodes per blk) ", INOPB);
			super1.s_isize = (unsigned)getnum((long)super0.s_isize - 2) + 2;
			if (super1.s_isize < i)
			{
				printf("Too small\n");
				continue;
			}
			if (super1.s_isize >= super1.s_fsize)
			{
				printf("Exceeds fsize!\n");
				continue;
			}
			if (super1.s_fsize - super1.s_isize >= usedb)
				break;
			printf("Insufficient space for files\n");
		}

		if ((i = fork()) > 0)
		{
			printf("Child continuing dcopy (pid %d)\n", i);
			exit(0);
		}
		if (i == 0)
		{
			signal(SIGINT, SIG_IGN);
			signal(SIGQUIT, SIG_IGN);
		}
		else
			printf("Fork has failed!! Parent will carry on...\n");
	}
	else		/* can't ask him so assume no change */
	{
		super1.s_fsize = super0.s_fsize;
		super1.s_isize = super0.s_isize;
	}
	super1.s_time = super0.s_time;

	/*
	 * compute the interleaving template
	 */
	icount = 0;
	for (i = 0; i < NSEC; i++)
	{
		while (flag[icount])
			icount = (icount + 1) % NSEC;
		adr[i] = icount;
		flag[icount]++;
		icount = (icount + IN) % NSEC;
	}

	{
		long fred;

		fred = (long)(super0.s_isize - 2);
		fred = fred * sizeof(short) * INOPB;
		i = (int)sbrk(0);
		fred += (long)(unsigned)i;
#ifdef	vax
		if ((int)brk((char *)fred) == -1)
#endif
#ifdef	pdp11
		if (fred >= 0177701L ||		/* pdp-unix sets this to 0 */
			(int)brk((char *)fred) == -1)
#endif
		{
			printf("Dcopy: Not enough core!\n");
			exit(1);
		}
	}
	indx = (short *)i;
	iindex = indx - 1;		/* as inodes start at one */

	/*
	 * scan ilist - copy files
	 */
	icount = (super0.s_isize - 2) * INOPB;
	fx = fi;
	i = icount;
	do
	{
		geti();
		if (node0.di_mode & IFMT)
		{
			copyi();
			*indx++ = inext++;	/* remember its new identity */
		}
		else *indx++ = 0;
	} while (--i);

	ipurge();		/* flush the remainder of the new ilist */
	iblk = 1;
	inum = INOPB;
	purgepool();		/* as about to read them back ... */
	fx = fo;		/* ... from the output device */

	/*
	 * fix up directory inode-references
	 */
	for (i = 1; i < inext; i++)
	{
		geti();
		if ((node0.di_mode & IFMT) == IFDIR)
			reldir();
	}

	/*
	 * fix up the free inode list in the new superblock
	 */
	icount = (super1.s_isize - 2) * INOPB;
	super1.s_tinode = icount - (inext - 1);
	while (i <= icount && super1.s_ninode != NICINOD)
		super1.s_inode[super1.s_ninode++] = i++;

	/*
	 * fix up the free block list
	 */
	i = BSIZE / (sizeof(short));
	indx = (short *)&in[0];
	do
		*indx++ = 0;
	while (--i);
	freelist();

	putblk((int *)&super1, SUPERB);
	purgepool();
	sync();
	printf("\007Dcopy finished\n");
	exit(0);
}

/*
 * read a number - check for legality; if \n pass back argument
 */
long
getnum(dflt)
long dflt;
{
	register c;
	register long n;

	for (;;)
	{
		n = 0;
		if ((c = getchar()) == EOF || c == '\n')
			return(dflt);
		while (c >= '0' && c <= '9')
		{
			n = n * 10L + (long)(c - '0');
			c = getchar();
		}
		if (c == EOF || c == '\n')
			return(n);
		while ((c = getchar()) != EOF && c != '\n');
		printf("\007Illegal number - try again ");
	}
}

getblk(f, b, n)
char *b;
daddr_t n;
{
	lseek(f, n << BSHIFT, 0);
	if (read(f, b, BSIZE) != BSIZE)
	{
		printf("Input error: block %ld\n", n);
		abort(0);
	}
}

putblk(b, n)
register int *b;
daddr_t n;
{
	register int *w, q;

	if (inpool == maxpool)
		purgepool();
	bpl[inpool].bn = n;
	w = (int *)(bpl[inpool++].bb);
	q = BSIZE / sizeof(int);
	do
		*w++ = *b++;
	while (--q);
}

purgepool()
{
	register int i;

	for (i = 0; i < inpool; i++)
	{
		lseek(fo, bpl[i].bn << BSHIFT, 0);
		if (write(fo, (char *)bpl[i].bb, BSIZE) != BSIZE)
		{
			printf("Output error: block %ld\n", bpl[i].bn);
			abort(1);
		}
	}
	inpool = 0;
}

geti()
{
	char *decnum();

	if (inum == INOPB)
	{
		getblk(fx, (char *)in_node, ++iblk);
		decnum(argv0, iblk);
		inum = 0;
	}
	node0 = in_node[inum++];
}

/*
 * overwrite argv[0] so that a `ps' shows what we are up to (yuk!)
 */
char *
decnum(s, n)
register char *s;
register daddr_t n;
{
	if ((n / 10) != 0)
		s = decnum(s, n / 10);
	*s++ = (char)(n % 10) + '0';
	*s = 0;
	return(s);
}

puti()
{
	out_node[onum++] = node1;
	if (onum == INOPB)
	{
		putblk((int *)out_node, oblk++);
		onum = 0;
	}
}

ipurge()
{
	register int *r, q;

	if (onum)
	{
		r = (int *)&out_node[onum];
		q = (sizeof(struct dinode)/sizeof(int)) * (INOPB - onum);
		do
			*r++ = 0;
		while (--q);
		putblk((int *)out_node, oblk++);
	}

	r = (int *)out_node;
	q = BSIZE / sizeof(int);
	do
		*r++ = 0;
	while (--q);

	while (oblk < super1.s_isize)
		putblk((int *)out_node, oblk++);

}

daddr_t
nxtblk()
{
	static int sector, track, cyl;
	register daddr_t block;

	for (;;)
	{
		block = cyl * NTRACK + track;
		block = block * NSEC + (daddr_t)adr[sector];
		lastblock = block;
		if (++sector >= NSEC)
		{
			sector = 0;
			if (++track >= NTRACK)
			{
				cyl++;
				track = 0;
			}
		}
		if (block >= super1.s_isize)
			return(block);
	}
}

freelist()
{
	register sector, track, cyl;
	daddr_t block;

	cyl = super1.s_fsize / (NTRACK * NSEC);
	sector = super1.s_fsize % (NTRACK * NSEC);
	track = sector / NSEC;		/* first non-usable block */
	sector = sector % NSEC;
	bfree((daddr_t)0);		/* end-of-list sentinel */

	while (sector)
	{
		block = cyl * NTRACK + track;
		block = block * NSEC + (daddr_t)--sector;
		bfree(block);
	}
	for (;;)
	{
		if (sector-- == 0)
		{
			sector += NSEC;
			if (track-- == 0)
			{
				track += NTRACK;
				cyl--;
			}
		}
		block = cyl * NTRACK + track;
		block = block * NSEC + (daddr_t)adr[sector];
		if (block == lastblock)		/* the last used block */
			return;
		if (block >= super1.s_isize)
			bfree(block);
	}
}

bfree(b)
daddr_t b;
{
	register i;

	if (super1.s_nfree == NICFREE)
	{
		((struct fblk *)in)->df_nfree = NICFREE;
		for (i = 0; i < NICFREE; i++)
			((struct fblk *)in)->df_free[i] = super1.s_free[i];
		putblk((int *)in, b);
		super1.s_nfree = 0;
	}
	super1.s_free[super1.s_nfree++] = b;
	if (b)
		super1.s_tfree++;
}

copyi()
{
	register int i, isdir;
	daddr_t doindr();

	node1 = node0;
	isdir = 0;
	switch(node1.di_mode & IFMT)
	{
	default:
		return;
	case IFIFO:
	case IFCHR:
	case IFBLK:
		break;
	case IFDIR:
		isdir++;
		dirsiz = 0;
		dbuf[0].d_nleft = 10;		/* 10 direct blocks */
		dbuf[0].d_upto = dbuf[0].d_base;
		dlevel = 0;
		dnum = DIRPB;
	case IFREG:
	case IFLOK:
	case IFALK:
		l3tol(bno, node1.di_addr, DNADDR);
		for (i = 0; i < DNADDR-3; i++)	/* direct blocks */
		{
			if (bno[i])
			{
				getblk(fi, in, bno[i]);
				if (isdir)
					copydir(0);
				else
				{
					bno[i] = nxtblk();
					putblk((int *)in, bno[i]);
				}
			}
		}
		if (bno[i])			/* first indirect */
			if (isdir)
				copydir(1);
			else
				bno[i] = doindr(bno[i], 1);
		i++;
		if (bno[i])			/* second indirect */
			if (isdir)
				copydir(2);
			else
				bno[i] = doindr(bno[i], 2);
		i++;
		if (bno[i])			/* third indirect */
			if (isdir)
				copydir(3);
			else
				bno[i] = doindr(bno[i], 3);
		if (isdir)
			flushdir();
		ltol3(node1.di_addr, bno, DNADDR);
		break;
	}
	puti();
}

daddr_t
doindr(b, level)
daddr_t b;
{
	register i;
	register daddr_t nb;
	daddr_t ind[NINDIR];

	getblk(fi, (char *)ind, b);
	nb = nxtblk();
	if (--level)
	{
		for (i = 0; i < NINDIR; i++)
			if (ind[i])
				ind[i] = doindr(ind[i], level);
	}
	else
	{
		for (i = 0; i < NINDIR; i++)
			if (ind[i])
			{
				getblk(fi, in, ind[i]);
				ind[i] = nxtblk();
				putblk((int *)in, ind[i]);
			}
	}
	putblk((int *)ind, nb);
	return(nb);
}

copydir(level)
{
	register struct direct *dp;
	register i;
	struct direct *dirnext();

	if (level)			/* an indirect block */
	{
		daddr_t ind[NINDIR];

		getblk(fi, (char *)ind, bno[level + 9]);
		if (--level)
		{
			for (i = 0; i < NINDIR; i++)
				if (ind[i])
					copydir(level);
		}
		else
		{
			for (i = 0; i < NINDIR; i++)
				if (ind[i])
				{
					getblk(fi, in, ind[i]);
					copydir(0);
				}
		}
		return;
	}
	/*
	 * at this point, have a data block in `in'
	 */
	i = DIRPB;
	dp = (struct direct *)in;
	do
	{
		if (dp->d_ino)
		{
			if (dnum == DIRPB)
			{
				dnum = 0;
				dirblk = dirnext();
			}
			dirblk[dnum++] = *dp;	/* copy entry */
			dirsiz += sizeof(struct direct);
		}
		dp++;
	} while (--i);
}

/*
 * allocate another directory data block
 */
struct direct *
dirnext()
{
	register i;
	register daddr_t nb;

	i = dlevel;
	nb = nxtblk();
	while (i && dbuf[i].d_nleft == 0)	/* an indirect block has been filled */
	{
		putblk((int *)dbuf[i].d_base, dbuf[i].d_nr);
		maxpool++;
		i--;
	}
	if (dbuf[i].d_nleft == 0)	/* time for next level of indirection */
		dlevel++;
	while (i < dlevel)	/* allocate indirect blocks */
	{
		*dbuf[i].d_upto++ = nb;
		if (i)
			dbuf[i].d_nleft--;
		i++;
		if (inpool == maxpool)
			purgepool();
		maxpool--;
		dbuf[i].d_nr = nb;
		dbuf[i].d_upto = dbuf[i].d_base;
		dbuf[i].d_nleft = NINDIR;
		nb = nxtblk();
	}
	dbuf[i].d_nleft--;
	*dbuf[i].d_upto++ = nb;
	if (inpool == maxpool)
		purgepool();
	bpl[inpool].bn = nb;
	return((struct direct *)bpl[inpool++].bb);
}

flushdir()
{
	register i;
	register int *r;

	/*
	 * clean up the last directory data block
	 */
	if (dnum != DIRPB)
	{
		r = (int *)&dirblk[dnum];
		i = (sizeof(struct direct)/sizeof(int)) * (DIRPB - dnum);
		do
			*r++ = 0;
		while (--i);
	}
	/*
	 * clean up indirect blocks
	 */
	for (i = dlevel; i < 3; i++)
		bno[i + 10] = 0;
	while (dlevel >= 0)
	{
		i = dbuf[dlevel].d_nleft;
		if (i)
			do
				*dbuf[dlevel].d_upto++ = 0;
			while (--i);
		if (dlevel--)
		{
			putblk((int *)dbuf[dlevel].d_base, dbuf[dlevel].d_nr);
			maxpool++;
		}
	}
	/*
	 * fix up size
	 */
	node1.di_size = dirsiz;
}

reldir()
{
	register i;

	l3tol(bno, node0.di_addr, DNADDR);
	for (i = 0; i < DNADDR-3; i++)
		if (bno[i])
			relx(bno[i]);
	if (bno[i])				/* first indirect */
		doirel(bno[i], 1);
	i++;
	if (bno[i])				/* second indirect */
		doirel(bno[i], 2);
	i++;
	if (bno[i])				/* third indirect */
		doirel(bno[i], 3);
	ltol3(node0.di_addr, bno, DNADDR);
}

doirel(b, level)
daddr_t b;
{
	register i;
	daddr_t ind[NINDIR];

	getblk(fo, (char *)ind, b);
	if (--level)
	{
		for (i = 0; i < NINDIR; i++)
			if (ind[i])
				doirel(ind[i], level);
	}
	else
	{
		for (i = 0; i < NINDIR; i++)
			if (ind[i])
				relx(ind[i]);
	}
}

relx(b)
daddr_t b;
{
	register struct  direct *dp;
	register int i;

	getblk(fo, in, b);
	dp = (struct direct *)in;
	i = DIRPB;
	do
	{
		if (dp->d_ino)
			dp->d_ino = iindex[dp->d_ino];
		dp++;
	} while (--i);
	putblk((int *)in, b);
}

/*
 * return how many blocks needed for inodes that are in use
 */
counti()
{
	register i, j, n;

	n = 0;
	for (i = 2; i < super0.s_isize; i++)
	{
		getblk(fi, (char *)&in_node[0], (daddr_t)i);
		for (j = 0; j < INOPB; j++)
			if (in_node[j].di_mode & IFMT)
				n++;
	}
	return((n + INOPB - 1) / INOPB);
}

daddr_t
countfree()
{
	register daddr_t m, n;

	n = (daddr_t)super0.s_nfree;
	if (n)
	{
		m = super0.s_free[0];
		while (m)
		{
			getblk(fi, in, m);
			n += (daddr_t)((struct fblk *)in)->df_nfree;
			m = ((struct fblk *)in)->df_free[0];
		}
	}
	return(n);
}
