/*
 * general TTY subroutines
 */
#include <param.h>
#include <systm.h>
#include <dir.h>
#include <signal.h>
#include <user.h>
#include <tty.h>
#include <sgtty.h>
#include <proc.h>
#include <inode.h>
#include <file.h>
#include <reg.h>
#include <conf.h>
#include <buf.h>
#include <lnode.h>
#include <errno.h>

extern struct user u;
extern char canonb[];
char	partab[];


/*
 * Input mapping table-- if an entry is non-zero, when the
 * corresponding character is typed preceded by "\" the escape
 * sequence is replaced by the table value.  Mostly used for
 * upper-case only terminals.
 */

char	maptab[] =
{
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,'|',000,000,000,000,000,'`',
	'{','}',000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,'~',000,
	000,'A','B','C','D','E','F','G',
	'H','I','J','K','L','M','N','O',
	'P','Q','R','S','T','U','V','W',
	'X','Y','Z',000,000,000,000,000,
};



/*
 * routine called on first teletype open.
 * establishes a process group for distribution
 * of quits and interrupts from the tty.
 */
ttyopen(dev, tp)
dev_t dev;
register struct tty *tp;
{
	register struct proc *pp;

	pp = u.u_procp;
	if ((pp->p_pid == pp->p_pgrp) &&
	    (u.u_ttyp == NULL) && (tp->t_pgrp == 0))
	{
		u.u_ttyp = tp;
		u.u_ttyd = dev;
		tp->t_pgrp = pp->p_pgrp;
	}
	tp->t_state &= ~WOPEN;
	tp->t_state |= ISOPEN;
}

/*
 * clean tp on last close
 */
ttyclose(tp)
register struct tty *tp;
{

	tp->t_pgrp = 0;
	wflushtty(tp);
	tp->t_state = 0;
}

/*
 * stty/gtty writearound
 */
stty()
{
	u.u_arg[2] = u.u_arg[1];
	u.u_arg[1] = TIOCSETP;
	ioctl();
}

gtty()
{
	u.u_arg[2] = u.u_arg[1];
	u.u_arg[1] = TIOCGETP;
	ioctl();
}

/*
 * ioctl system call
 * Check legality, execute common code, and switch out to individual
 * device routine.
 */
ioctl()
{
	register struct file *fp;
	register struct inode *ip;
	register struct a
	{
		int	fdes;
		int	cmd;
		caddr_t	cmarg;
	} *uap;
	register dev_t dev;
	register fmt;

	uap = (struct a *)u.u_ap;
	if ((fp = getf(uap->fdes)) == NULL)
		return;
	ip = fp->f_inode;
	fmt = ip->i_mode & IFMT;
	if (fmt != IFCHR)
	{
		u.u_error = ENOTTY;
		return;
	}
	dev = getdevno(ip);
	u.u_segflg = USERD;		/* ttioccom may use iomove */
	(*cdevsw[major(dev)].d_ioctl)(dev, uap->cmd, uap->cmarg, fp->f_flag);
}

/*
 * Common code for several tty ioctl commands
 */
ttioccomm(com, tp, addr, dev)
register struct tty *tp;
caddr_t addr;
{
	unsigned t;
	struct sgttyb iocb;
	extern int nldisp;

	switch(com)
	{

	/*
	 * prevent more opens on channel
	 */
	case TIOCEXCL:
		tp->t_state |= XCLUDE;
		break;

	case TIOCNXCL:
		tp->t_state &= ~XCLUDE;
		break;

	/*
	 * Set new parameters
	 */
	case TIOCSETP:
		wflushtty(tp);
	case TIOCSETN:
		/*
		 * As this routine may be called from the kernel (mx driver),
		 * iomove must be used to collect arguments instead of copyin.
		 * u_segflg has already been set up correctly.
		 */
		u.u_base = addr;
		iomove((caddr_t)&iocb, sizeof(iocb), B_WRITE);
		if (u.u_error)
			return(1);
		if ((iocb.sg_ispeed < tp->t_ispeed && iocb.sg_ospeed < tp->t_ospeed)
		    || u.u_uid == 0
		    || (u.u_procp->p_lnode->l_flags & TTYOPTS))
		{
			tp->t_ispeed = iocb.sg_ispeed;
			tp->t_ospeed = iocb.sg_ospeed;
		}
		tp->t_erase = iocb.sg_erase;
		tp->t_kill = iocb.sg_kill;
		tp->t_flags = iocb.sg_flags;
		break;

	/*
	 * send current parameters to user
	 */
	case TIOCGETP:
		iocb.sg_ispeed = tp->t_ispeed;
		iocb.sg_ospeed = tp->t_ospeed;
		iocb.sg_erase = tp->t_erase;
		iocb.sg_kill = tp->t_kill;
		iocb.sg_flags = tp->t_flags;
		if (copyout((caddr_t)&iocb, addr, sizeof(iocb)))
			u.u_error = EFAULT;
		break;

	/*
	 * Hang up line on last close
	 */

	case TIOCHPCL:
		tp->t_state |= HUPCLS;
		break;

	case TIOCFLUSH:
		flushtty(tp);
		break;

	default:
		return(0);
	}
	return(1);
}

/*
 * Wait for output to drain, then flush input waiting.
 */
wflushtty(tp)
register struct tty *tp;
{
	spl5();
	while (tp->t_outq.c_cc && tp->t_state&CARR_ON)
	{
		(*tp->t_oproc)(tp);
		tp->t_state |= ASLEEP;
		sleep((caddr_t)&tp->t_outq, TTOPRI);
	}
	flushtty(tp);
	spl0();
}

/*
 * flush all TTY queues
 */
flushtty(tp)
register struct tty *tp;
{
	register s;

	while (getc(&tp->t_canq) >= 0)
		;
	wakeup((caddr_t)&tp->t_rawq);
	wakeup((caddr_t)&tp->t_outq);
	s = spl6();
	tp->t_state &= ~TTSTOP;
	while (getc(&tp->t_outq) >= 0)
		;
	while (getc(&tp->t_rawq) >= 0)
		;
	tp->t_delct = 0;
	splx(s);
}



/*
 * transfer raw input list to canonical list,
 * doing erase-kill processing and handling escapes.
 * It waits until a full line has been typed in cooked mode,
 * or until any character has been typed in raw mode.
 */
canon(tp)
register struct tty *tp;
{
	register char *bp;
	char *bp1;
	register int c;
	int mc;
	int slosh;

	spl5();
	while (tp->t_delct == 0)
	{
		if ((tp->t_state & CARR_ON) == 0)
		{
			spl0();
			return(0);
		}
		sleep((caddr_t)&tp->t_rawq, TTIPRI);
	}
	spl0();
	bp = canonb;
	slosh = 0;
	while ((c = getc(&tp->t_rawq)) >= 0)
	{
		if (c == 0377)
		{
			tp->t_delct--;
			break;
		}
		if (slosh)
		{
			slosh = 0;
			mc = maptab[c];
			if (mc && (tp->t_flags & LCASE))
			{
				c = mc;
				bp--;
			}
			else if (c == tp->t_erase || c == tp->t_kill)
				bp--;
		}
		else
		{
			if (c == tp->t_erase)
			{
				if (bp > canonb)
					bp--;
				continue;
			}
			if (c == tp->t_kill)
				c = 0377;
			else if (c == CEOT)
				continue;
			if (c == '\\')
				slosh = 1;
		}
		*bp++ = c;
		if (bp >= canonb + CANBSIZ)
			break;
	}
	bp1 = canonb;
	c = bp1;
	while (bp1 < bp)
		if (*bp1++ == 0177777)
			c = bp1;
	bp1 = c;
	b_to_q(bp1, bp-bp1, &tp->t_canq);

	if (tp->t_state&TBLOCK && tp->t_rawq.c_cc < TTYHOG/5)
	{
		if (putc(CSTART, &tp->t_outq)==0)
		{
			tp->t_state &= ~TBLOCK;
			ttstart(tp);
		}
		tp->t_char = 0;
	}

	return(bp-bp1);
}


/*
 * Place a character on raw TTY input queue, putting in delimiters
 * and waking up top half as needed.
 * Also echo if required.
 * The arguments are the character and the appropriate
 * tty structure.
 */
ttyinput(c, tp)
register c;
register struct tty *tp;
{
	register int t_flags;

	c &= 0377;
	if (tp->t_rtype)		/* redirection ? */
	{
		if (tp->t_rtype & TMASTER && (c & 0177) == tp->t_cchar)
		{
			tp->t_rtype = 0;
			tp->t_cchar = 0;
			tp->t_redirect->t_rtype = 0;
			tp->t_redirect = tp->t_redirect->t_redirect = NULL;
			wakeup((caddr_t)&tp->t_rtype);
			return;
		}
		switch(tp->t_rtype)
		{
		case TMASTER+TCONLOG:
		case TCONLOG:
			tp = tp->t_redirect;
			if (tp->t_outq.c_cc < TTYHOG)
			{
				putc(c & 0177, &tp->t_outq);	/* quick and dirty */
				ttstart(tp);
			}
			return;

		case TMASTER+TCONSHARE:
		case TMASTER+TCONGRAB:
			tp = tp->t_redirect;
			break;

		case TCONGRAB:
			return;

		case TMULTIPLEX:
			mxinput(c, tp);
			return;

		}
	}
	t_flags = tp->t_flags;
	if ((t_flags&RAW)==0)
	{
		c &= 0177;
		if (tp->t_state&TTSTOP)
		{
			tp->t_state &= ~TTSTOP;
			ttstart(tp);
			if (c == CSTOP || c == CSTART)
				return;
		}
		else if (c==CSTOP)
		{
			tp->t_state |= TTSTOP;
			return;
		}
		if (c==CQUIT || c==CINTR)
		{
			flushtty(tp);
			c = (c==CINTR) ? SIGINT:SIGQUIT;
			signal(tp->t_pgrp, c);
			return;
		}
		if (c=='\r' && t_flags&CRMOD)
			c = '\n';
		if (t_flags&LCASE && c>='A' && c<='Z')
			c += 'a'-'A';
	}
	if (tp->t_rawq.c_cc>TTYHOG)
	{
		flushtty(tp);	/* Note: the TTYHOG'th char is not echoed */
		return;		/* indicating the queue has been flushed */
	}
	putc(c, &tp->t_rawq);
	if (t_flags & (RAW | CBREAK))
	{
		if (tp->t_state & RAWSLEEP)
		{
			tp->t_state &= ~RAWSLEEP;
			wakeup((caddr_t)&tp->t_rawq);
		}
	}
	else if (c == '\n' || c == CEOT || c == CBRK)
	{
		if (putc(0377, &tp->t_rawq) == 0)
			tp->t_delct++;
		wakeup((caddr_t)&tp->t_rawq);
	}
	if (t_flags&ECHO)
	{
		ttyoutput(c, tp);
		if (c=='\b' && tp->t_erase=='\b' && (t_flags&(RAW|CBREAK))==0)
		{
			ttyoutput(' ', tp);
			ttyoutput(c, tp);
		}
		ttstart(tp);
	}
	else if (tp->t_rtype == TCONVIEW || tp->t_rtype == TCONSHARE)
	{
		tp = tp->t_redirect;
		ttyoutput(c, tp);
		ttstart(tp);
	}
	if (tp->t_rawq.c_cc > TTYHOG - TTYBELL)	/* queue getting full */
		ttyoutput('\07', tp);
}


/*
 * put character on TTY output queue, adding delays,
 * expanding tabs, and handling the CR/NL bit.
 * It is called both from the top half for output, and from
 * interrupt level for echoing.
 * The arguments are the character and the tty structure.
 */
ttyoutput(c, tp)
register c;
register struct tty *tp;
{
	register char *colp;
	register ctype;

	switch(tp->t_rtype)		/* redirection? */
	{
	case TCONVIEW:
	case TCONSHARE:
		ttyoutput(c, tp->t_redirect);
		colp = (char *)spl5();
		ttstart(tp->t_redirect);
		splx(colp);
		break;

	case TCONGRAB:
		tp = tp->t_redirect;
		ttyoutput(c, tp);
		c = spl5();
		ttstart(tp);
		splx(c);
		return;
	}

	/*
	 * Ignore EOT in normal mode to avoid hanging up
	 * certain terminals.
	 * In raw mode dump the char unchanged.
	 */

	if (tp->t_flags & RAW)
	{
		putc(c, &tp->t_outq);
		return;
	}
	c &= 0177;
	if (c==CEOT)
		return;

	/*
	 * Turn tabs to spaces as required
	 */
	if (c=='\t' && (tp->t_flags&XTABS))
	{
		c = 8;
		do
			ttyoutput(' ', tp);
		while (--c >= 0 && tp->t_col&07);
		return;
	}
	/*
	 * for upper-case-only terminals,
	 * generate escapes.
	 */
	if (tp->t_flags&LCASE)
	{
		colp = "({)}!|^~'`";
		while(*colp++)
			if(c == *colp++)
			{
				ttyoutput('\\', tp);
				c = colp[-2];
				break;
			}
		if ('a'<=c && c<='z')
			c += 'A' - 'a';
	}
	/*
	 * turn <nl> to <cr><lf> if desired.
	 */
	if (c=='\n' && tp->t_flags&CRMOD)
		ttyoutput('\r', tp);
	putc(c, &tp->t_outq);
	/*
	 * Calculate delays.
	 * The numbers here represent clock ticks
	 * and are not necessarily optimal for all terminals.
	 * The delays are indicated by characters above 0200.
	 * In raw mode there are no delays and the
	 * transmission path is 8 bits wide.
	 */
	colp = &tp->t_col;
	ctype = partab[c];
	c = 0;
	switch (ctype & 077)
	{

	/* ordinary */
	case 0:
		(*colp)++;

	/* non-printing */
	case 1:
		break;

	/* backspace */
	case 2:
		if (*colp)
			(*colp)--;
		break;

	/* newline */
	case 3:
		ctype = (tp->t_flags >> 8) & 03;
		if(ctype == 1) /* tty 37 */
		{
			if (*colp)
				c = max(((unsigned)*colp>>4) + 3, (unsigned)6);
		} else
		if(ctype == 2) /* vt05 */
		{
			c = 6;
		}
		*colp = 0;
		break;

	/* tab */
	case 4:
		ctype = (tp->t_flags >> 10) & 03;
		if(ctype == 1) /* tty 37 */
		{
			c = 1 - (*colp | ~07);
			if(c < 5)
				c = 0;
		}
		*colp |= 07;
		(*colp)++;
		break;

	/* vertical motion */
	case 5:
		if(tp->t_flags & VTDELAY) /* tty 37 */
			c = 0177;
		break;

	/* carriage return */
	case 6:
		ctype = (tp->t_flags >> 12) & 03;
		if(ctype == 1) /* tn 300 */
		{
			c = 5;
		} else if(ctype == 2) /* ti 700 */
		{
			c = 10;
		}
		*colp = 0;
	}
	if(c)
		putc(c|0200, &tp->t_outq);
}

/*
 * Restart typewriter output following a delay
 * timeout.
 * The name of the routine is passed to the timeout
 * subroutine and it is called during a clock interrupt.
 */
ttrstrt(tp)
register struct tty *tp;
{

	tp->t_state &= ~TIMEOUT;
	ttstart(tp);
}

/*
 * Start output on the typewriter. It is used from the top half
 * after some characters have been put on the output queue,
 * from the interrupt routine to transmit the next
 * character, and after a timeout has finished.
 */
ttstart(tp)
register struct tty *tp;
{
	register s;

	s = spl5();
	if((tp->t_state&(TIMEOUT|TTSTOP|BUSY)) == 0)
		(*tp->t_oproc)(tp);
	splx(s);
}

/*
 * Called from device's read routine after it has
 * calculated the tty-structure given as argument.
 */
ttread(tp)
register struct tty *tp;
{
	if (tp->t_flags & (RAW|CBREAK))
	{
		do
		{
			spl5();
			while (tp->t_rawq.c_cc == 0)
				if (tp->t_state & CARR_ON)
				{
					tp->t_state |= RAWSLEEP;
					sleep(&tp->t_rawq, TTIPRI);
				}
				else
				{
					spl0();
					return(0);
				}
			spl0();
		} while (passc(getc(&tp->t_rawq)) >= 0);
		return(0);
	}
	if ((tp->t_state&CARR_ON)==0)
		return(0);
	if (tp->t_canq.c_cc || canon(tp))
		while (tp->t_canq.c_cc && passc(getc(&tp->t_canq))>=0)
			;
	return(tp->t_rawq.c_cc + tp->t_canq.c_cc);
}

/*
 * Called from the device's write routine after it has
 * calculated the tty-structure given as argument.
 */
caddr_t
ttwrite(tp)
register struct tty *tp;
{
	register c;

	if ((tp->t_state&CARR_ON)==0)
		return(NULL);
	while (u.u_count)
	{
		spl5();
		while (tp->t_outq.c_cc > TTHIWAT)
		{
			ttstart(tp);
			tp->t_state |= ASLEEP;
			sleep((caddr_t)&tp->t_outq, TTOPRI);
		}
		spl0();
		if ((c = cpass()) < 0)
			break;
		ttyoutput(c, tp);
	}
	ttstart(tp);
	return(NULL);
}

struct tty *
ttyget(fd)
{
	register struct inode *ip;

	{
		register struct file *fp;

		if ((fp = getf(fd)) == NULL)
			return(NULL);
		if ((fp->f_flag & (FREAD|FWRITE)) != (FREAD|FWRITE))
		{
			u.u_error = EACCES;
			return(NULL);
		}
		ip = fp->f_inode;
	}
	if ((ip->i_mode & IFMT) != IFCHR)
	{
		u.u_error = ERDRCT;
		return(NULL);
	}
	{
		register struct tty *tp;
		register dev;

		dev = getdevno(ip);
		if ((tp = cdevsw[major(dev)].d_ttys) == NULL)
		{
			u.u_error = ERDRCT;
			return(NULL);
		}
		return(&tp[minor(dev)]);
	}
}

ttyconnect()
{
	register struct tty *mtp, *stp;
	register struct a
	{
		int fd1, fd2;
		int type;
		char cchar;
	} *uap;

	if (!suser())
		return;
	uap = (struct a *)u.u_ap;
	if ((mtp = ttyget(uap->fd1)) == NULL || (stp = ttyget(uap->fd2)) == NULL)
		return;
	spl5();
	switch(uap->type)
	{
	case TCONDIS:				/* disconnect */
		if (mtp != stp || mtp->t_rtype == 0 ||
		    mtp->t_rtype == TMULTIPLEX)
			u.u_error = ERDRCT;
		else
		{
			stp = mtp->t_redirect;
			if (mtp->t_rtype & TMASTER)
				wakeup((caddr_t)&mtp->t_rtype);
			else
				wakeup((caddr_t)&stp->t_rtype);
			mtp->t_rtype = stp->t_rtype = 0;
			mtp->t_cchar = 0;
			mtp->t_redirect = stp->t_redirect = NULL;
		}
		break;

	case TCONLOG:
	case TCONVIEW:
	case TCONGRAB:
	case TCONSHARE:
		if (mtp == stp || mtp->t_rtype || stp->t_rtype)
			u.u_error = ERDRCT;
		else
		{
			mtp->t_rtype = uap->type | TMASTER;
			stp->t_rtype = uap->type;
			mtp->t_cchar = uap->cchar;
			mtp->t_redirect = stp;
			stp->t_redirect = mtp;
			while (mtp->t_rtype)
				sleep((caddr_t)&mtp->t_rtype, PREDIR);
		}
		break;

	default:
		u.u_error = EINVAL;
		break;
	}
	spl0();
}

/*
 *	uprints - print a string on the controlling tty
 *	high water marks are ignored so don't be verbose
 */
uprints(s)
register char *s;
{
	register struct tty *tp;
	extern struct tty kl_tty;	/* console tty structure */

	if ((tp = u.u_ttyp) == NULL)
		tp = &kl_tty;
	while (*s)
		ttyoutput(*s++, tp);
	ttstart(tp);
}
